'use strict';

const { banDuration } = require('yandex-config');
const assert = require('helpers/assert');
const moment = require('moment');

const { Ban, TrialTemplate } = require('db/postgres');

const BaseModel = require('models/base');
const ExamModel = require('models/exam');

class BanModel extends BaseModel {
    static *banUser(banData, transaction) {
        const {
            globalUserId,
            adminId,
            reason,
            trialTemplateIds,
            isSuperban,
            userLogin
        } = banData;

        yield Ban.update(
            { isLast: false },
            {
                where: { globalUserId, trialTemplateId: trialTemplateIds },
                fields: ['isLast'],
                transaction
            }
        );

        const startedDate = new Date();
        let expiredDate;

        if (!isSuperban) {
            expiredDate = moment(startedDate).add(banDuration.count, banDuration.unit).startOf('day').toDate();
        }

        const dataForCreate = trialTemplateIds.map(trialTemplateId => {
            return {
                globalUserId,
                adminId,
                reason,
                action: 'ban',
                startedDate,
                expiredDate,
                trialTemplateId,
                userLogin,
                isLast: true
            };
        });

        try {
            yield Ban.bulkCreate(dataForCreate, { transaction });
        } catch (error) {
            const isOtherError = error.name !== 'SequelizeForeignKeyConstraintError';

            assert(isOtherError, 400, 'Foreign key constraint error', 'FCE',
                { message: error.message }
            );

            throw error;
        }
    }

    static *findLastBan(globalUserId, examIdentity, transaction) {
        const condition = ExamModel.getFindCondition(examIdentity);

        const includeTrialTemplate = {
            model: TrialTemplate,
            attributes: [],
            where: condition,
            as: 'trialTemplate'
        };

        return yield Ban.findOne({
            where: {
                globalUserId,
                isLast: true,
                action: 'ban'
            },
            attributes: ['expiredDate'],
            include: [includeTrialTemplate],
            transaction
        });
    }

    static *unbanUser(unbanData, transaction) {
        const {
            globalUserId,
            adminId,
            reason,
            trialTemplateIds,
            userLogin
        } = unbanData;

        yield Ban.update(
            { isLast: false },
            {
                where: { globalUserId, trialTemplateId: trialTemplateIds },
                fields: ['isLast'],
                transaction
            }
        );

        const dataForCreate = trialTemplateIds.map(trialTemplateId => {
            return {
                globalUserId,
                adminId,
                reason,
                action: 'unban',
                startedDate: new Date(),
                trialTemplateId,
                userLogin,
                isLast: true
            };
        });

        try {
            yield Ban.bulkCreate(dataForCreate, { transaction });
        } catch (error) {
            const isOtherError = error.name !== 'SequelizeForeignKeyConstraintError';

            assert(isOtherError, 400, 'Foreign key constraint error', 'FCE',
                { message: error.message }
            );

            throw error;
        }
    }

}

module.exports = BanModel;
