'use strict';

const config = require('yandex-config');
const { chunkSize } = config.blackbox;
const log = require('logger');
const _ = require('lodash');
const ip = require('ip');
const getTvmTicket = require('helpers/tvm').getTicket;
const yabox = require('yabox');

class Blackbox {
    constructor(blackboxConfig) {
        this._yabox = yabox(blackboxConfig.connection);
        this._attributes = blackboxConfig.attributes;
        this._tvmName = blackboxConfig.tvmName;
    }

    *sessionId(options) {
        options = _.assign({
            attributes: this._attributes,
            emails: 'getdefault'
        }, options);

        return yield this._requestToBlackbox('sessionid', options);
    }

    *userInfo(options) {
        options = _.assign({
            attributes: this._attributes
        }, options);

        return yield this._requestToBlackbox('userinfo', options);
    }

    *_requestToBlackbox(method, options) {
        options = yield this._getOptionsWithTvm(options);

        const beginRequest = new Date();
        const res = yield this._yabox[method](options);
        const requestTime = new Date() - beginRequest;

        if (requestTime > config.warnLogTimeout) {
            options.requestTime = requestTime;
            log.warn('Request to Blackbox is slowly', options);
        }

        return res.body;
    }

    *getEmails(uids) {
        const usersInfo = yield _
            .chunk(uids, chunkSize)
            .map(this._getUsersInfo.bind(this));

        return _(usersInfo)
            .flatten()
            .transform((result, item) => {
                const uid = _.get(item, 'uid.value');

                result[uid] = _.get(item, 'address-list.0.address', '');
            }, {})
            .value();
    }

    *_getUsersInfo(uids) {
        const blackboxData = yield this.userInfo({
            uid: uids.join(','),
            userip: ip.address(),
            emails: 'getdefault',
            format: 'json'
        });

        return _.get(blackboxData, 'users', []);
    }

    *_getOptionsWithTvm(options) {
        const { forceTvmCheck } = options;
        const newOptions = _.assign({}, options);

        delete newOptions.forceTvmCheck;

        if (config.skipTvmCheck && !forceTvmCheck) {
            return newOptions;
        }

        const tvmTicket = yield getTvmTicket(config.tvm[this._tvmName]);

        return _.assign(newOptions, {
            headers: { 'x-ya-service-ticket': tvmTicket }
        });
    }
}

module.exports = Blackbox;
