'use strict';

const BaseModel = require('models/base');
const DrawModel = require('models/draw');
const MdsModel = require('models/mds');
const Notifier = require('models/notifier');
const WebUser = require('models/user/webUser');
const ProctoringResponse = require('models/proctoringResponse');
const QRcodeController = require('models/draw/qr_code');

const assert = require('helpers/assert');
const date = require('helpers/dateHelper');
const getTvmTicket = require('helpers/tvm').getTicket;

const _ = require('lodash');
const got = require('got');
const co = require('co');
const log = require('logger');
const config = require('yandex-config');

const {
    Certificate,
    Trial,
    TrialTemplate,
    Service,
    Type,
    User,
    AuthType
} = require('db/postgres');

const CERT_FIELDS = [
    'confirmedDate',
    'dueDate',
    'firstname',
    'lastname',
    'active',
    'imagePath'
];
const USER_FIELDS = [
    'firstname',
    'lastname'
];
const FORMAT_TO_FIELD = {
    png: 'imagePath',
    pdf: 'pdfPath'
};

class CertificateModel extends BaseModel {

    static *_findById(certId) {
        const includeType = {
            model: Type,
            where: { code: 'cert' },
            attributes: ['code']
        };
        const includeService = { model: Service };
        const trialTemplateInclude = {
            model: TrialTemplate,
            attributes: ['id', 'previewImagePath', 'language', 'isProctoring'],
            include: [includeService],
            as: 'trialTemplate'
        };
        const includeTrial = {
            model: Trial,
            attributes: ['trialTemplateId', 'userId'],
            where: { nullified: 0 },
            include: trialTemplateInclude
        };

        const certData = yield Certificate.findOne({
            where: { id: certId },
            attributes: [
                'id',
                'confirmedDate',
                'dueDate',
                'firstname',
                'lastname',
                'active',
                'imagePath',
                'pdfPath'
            ],
            include: [includeTrial, includeType]
        });

        assert(certData, 404, 'Certificate not found', 'CNF');
        assert(certData.active === 1, 403, 'Certificate was nullified', 'CWN');

        return certData;
    }

    static *find(certId, lastname) {
        const certData = yield CertificateModel._findById(certId);
        const dbLastname = _.get(certData, 'lastname');
        const dbLowerLastname = _.toLower(dbLastname).trim();
        const lowerLastname = _.toLower(lastname).trim();
        const isCorrectLastname = lowerLastname === dbLowerLastname;

        assert(isCorrectLastname, 404, 'Certificate not found', 'CNF');

        return certData;
    }

    static *_selectDataForCreate(attempt, transaction) {
        const includeUser = {
            model: User,
            as: 'user',
            attributes: _.concat(USER_FIELDS, 'id')
        };
        const includeServices = {
            model: Service,
            as: 'service',
            attributes: ['code']
        };
        const includeType = {
            model: Type,
            attributes: ['id', 'code'],
            as: 'type'
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['validityPeriod', 'language', 'isProctoring'],
            include: [includeServices, includeType]
        };

        return yield Trial.findById(attempt.id, {
            include: [includeTrialTemplate, includeUser],
            attributes: ['id', 'finished', 'passed'],
            transaction
        });
    }

    static *tryCreate(attempt, data, transaction) {
        if (!attempt.get('passed') || (data.isProctoring && !data.isProctoringCorrect)) {
            return;
        }

        yield CertificateModel.create(attempt, transaction);
    }

    static *create(attempt, transaction) {
        const trial = yield CertificateModel._selectDataForCreate(attempt, transaction);
        const userData = _.pick(trial.user, USER_FIELDS);
        const confirmedDate = yield CertificateModel._getConfirmedDate(trial, transaction);
        const validityPeriod = _.get(trial, 'trialTemplate.validityPeriod');
        const calculateData = {
            dueDate: date.addToDate(confirmedDate, validityPeriod),
            trialId: attempt.id,
            active: 1,
            confirmed: 1,
            confirmedDate,
            typeId: _.get(trial, 'trialTemplate.type.id')
        };

        const cert = yield Certificate.create(_.assign(calculateData, userData), { transaction });
        const typeCode = _.get(trial, 'trialTemplate.type.code');

        yield CertificateModel._createPath({
            trial,
            certData: cert,
            format: 'png',
            typeCode
        }, transaction);
    }

    static *_getConfirmedDate(trial, transaction) {
        const proctoringResponse = yield ProctoringResponse.tryFindLast(trial.id, transaction);

        return proctoringResponse ? proctoringResponse.time : trial.finished;
    }

    static *_createPath(data, transaction) {
        const { format, certData, trial, typeCode } = data;

        const drawData = yield CertificateModel._getDrawData({ trial, certData, typeCode, format });

        const result = DrawModel.drawCert(drawData, format);

        return yield this._putAndSavePath({ result, certData, format }, transaction);
    }

    static *_getDrawData(data) {
        const certId = _.get(data, 'certData.id');
        const lastname = _.get(data, 'certData.lastname');

        const qrUrl = yield QRcodeController.createQRcodeUrl(certId, lastname, data.format);

        return _.assign(
            {
                qrUrl,
                service: _.get(data, 'trial.trialTemplate.service.code'),
                language: _.get(data, 'trial.trialTemplate.language'),
                type: _.get(data, 'typeCode'),
                certId,
                isProctoring: _.get(data, 'trial.trialTemplate.isProctoring')
            },
            _.pick(data.certData.toJSON(), ['dueDate', 'confirmedDate', 'firstname', 'lastname'])
        );
    }

    static *_putAndSavePath(data, transaction) {
        const { result, format, certData } = data;

        let path = result.url;

        if (!_.isString(path)) {
            path = yield MdsModel.putCert(result.buffer, certData.id, format);
        }

        certData.set(FORMAT_TO_FIELD[format], path);

        yield certData.save({ transaction });

        return path;
    }

    static *findAll(uid) {
        const data = yield {
            certsData: CertificateModel._getCertificates(uid),
            userInfo: WebUser.findUser({ where: { uid }, attributes: ['firstname', 'lastname'] })
        };

        assert(data.userInfo, 404, 'User not found', 'UNF');

        const certificates = yield CertificateModel._pickCertificatesData(data.certsData);

        const firstname = data.userInfo.get('firstname');
        const lastname = data.userInfo.get('lastname');

        return { firstname, lastname, certificates };
    }

    static *findCertsByUids(uids) {
        const certs = yield CertificateModel._getCertificates(uids);
        const now = new Date();
        const activeCerts = _.filter(certs, cert => cert.dueDate > now);

        const certsByUids = _.groupBy(activeCerts, 'trial.user.uid');

        yield uids.map(function *(uid) {
            const hasCerts = _.has(certsByUids, uid);

            certsByUids[uid] = hasCerts ? yield CertificateModel._pickCertificatesData(certsByUids[uid]) : [];
        });

        return certsByUids;
    }

    static *_pickCertificatesData(certsData) {
        return yield certsData.map(function *(certData) {
            const cert = yield CertificateModel.pickCertificateData(certData);

            CertificateModel._setExamInfo(certData, cert);

            return cert;
        });
    }

    static *pickCertificateData(certData) {
        const cert = certData.toJSON();
        const typeCode = _.get(cert, 'type.code', '');

        if (!cert.imagePath) {
            const dataForCreate = {
                trial: cert.trial,
                certData,
                format: 'png',
                typeCode
            };

            _.set(cert, 'imagePath', yield CertificateModel._createPath(dataForCreate));
        }

        return _(cert)
            .pick(CERT_FIELDS)
            .assign({
                certId: cert.id,
                imagePath: MdsModel.getAvatarsPath(cert.imagePath),
                service: _.get(cert, 'trial.trialTemplate.service'),
                certType: typeCode,
                previewImagePath: MdsModel.getAvatarsPath(_.get(cert, 'trial.trialTemplate.previewImagePath'))
            })
            .value();
    }

    static *_getCertificates(uids) {
        if (!_.isArray(uids)) {
            uids = [uids];
        }

        const includeServices = {
            model: Service,
            as: 'service'
        };
        const includeType = {
            model: Type,
            as: 'type',
            attributes: ['code']
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['slug', 'previewImagePath', 'language', 'isProctoring'],
            include: [includeServices]
        };
        const includeAuthType = {
            model: AuthType,
            attributes: [],
            where: { code: 'web' },
            as: 'authType'
        };
        const includeUser = {
            model: User,
            attributes: ['uid'],
            where: { uid: uids },
            include: includeAuthType
        };
        const includeTrial = {
            model: Trial,
            where: { nullified: 0 },
            attributes: ['trialTemplateId'],
            include: [includeTrialTemplate, includeUser]
        };

        return yield Certificate.findAll({
            where: { active: 1 },
            attributes: [
                'id',
                'confirmedDate',
                'dueDate',
                'firstname',
                'lastname',
                'active',
                'imagePath'
            ],
            include: [includeTrial, includeType],
            order: [['dueDate', 'DESC']]
        });
    }

    static *getMyCertificates(uid, login, getAttemptInfo) {
        const certsData = yield CertificateModel._getCertificates(uid);

        const states = yield _(certsData)
            .map(certData => _.get(certData, 'trial.trialTemplateId'))
            .uniq()
            .reduce((acc, trialTemplateId) => {
                acc[trialTemplateId] = getAttemptInfo(trialTemplateId, {
                    uid,
                    login,
                    authTypeCode: 'web'
                });

                return acc;
            }, {});

        return yield certsData.map(function *(certData) {
            const cert = yield CertificateModel.pickCertificateData(certData);
            const trialTemplateId = _.get(certData, 'trial.trialTemplateId');

            CertificateModel._setExamInfo(certData, cert);

            cert.check = states[trialTemplateId];

            return cert;
        });
    }

    static _setExamInfo(data, cert) {
        const trialTemplateId = _.get(data, 'trial.trialTemplateId');
        const slug = _.get(data, 'trial.trialTemplate.slug');

        _.set(cert, 'exam.slug', slug);
        _.set(cert, 'exam.id', trialTemplateId);
    }

    static *getPdf(certId, uid) {
        const data = yield {
            certData: CertificateModel._findById(certId),
            userInfo: WebUser.findUser({ where: { uid }, attributes: ['id'] })
        };
        const isCorrectUserId = _.get(data.certData, 'trial.userId') === _.get(data.userInfo, 'id');

        assert(isCorrectUserId, 404, 'Certificate not found', 'CNF');

        const { certData } = data;

        if (certData.pdfPath) {
            const pdfUrl = MdsModel.getMdsPath(certData.pdfPath);

            return got.stream(pdfUrl);
        }

        const typeCode = _.get(certData, 'type.code');
        const dataGetDrawData = { trial: certData.trial, certData, typeCode, format: 'pdf' };
        const drawData = yield CertificateModel._getDrawData(dataGetDrawData);

        const result = DrawModel.drawCert(drawData, 'pdf');

        co(this._putAndSavePath({ result, certData, format: 'pdf' }));

        return result.buffer;
    }

    static *nullify(certIds, deactivateReason, transaction) {
        yield Certificate.update(
            {
                active: 0,
                deactivateReason,
                deactivateDate: new Date()
            },
            {
                fields: ['active', 'deactivateReason', 'deactivateDate'],
                where: { id: certIds },
                transaction
            }
        );
    }

    static *getUsersForNullifyCerts(certIds, transaction) {
        const includeService = {
            model: Service,
            as: 'service',
            attributes: ['code']
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['language'],
            include: includeService
        };
        const includeCertificates = {
            model: Certificate,
            as: 'certificates',
            where: { id: certIds },
            attributes: []
        };
        const includeTrials = {
            model: Trial,
            as: 'trials',
            attributes: ['id'],
            include: [includeCertificates, includeTrialTemplate]
        };
        const users = yield User.findAll({
            attributes: ['uid'],
            include: includeTrials,
            order: [
                'uid',
                [includeTrials, includeCertificates, 'id']
            ],
            transaction
        });

        return _.map(users, user => ({
            uid: user.uid,
            services: this._getServicesByTrials(user.trials)
        }));
    }

    static _getServicesByTrials(trials) {
        return _.map(trials, trial => ({
            code: _.get(trial, 'trialTemplate.service.code'),
            language: _.get(trial, 'trialTemplate.language')
        }));
    }

    static *getCertsByGlobalUser(globalUserId, trialTemplateIds, transaction) {
        const includeService = {
            model: Service,
            attributes: ['code']
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            where: { id: trialTemplateIds },
            as: 'trialTemplate',
            attributes: ['language'],
            include: [includeService]
        };
        const includeUser = {
            model: User,
            where: { globalUserId },
            attributes: ['uid']
        };
        const includeTrial = {
            model: Trial,
            attributes: ['id'],
            include: [includeUser, includeTrialTemplate]
        };
        const certs = yield Certificate.findAll({
            where: { active: 1, dueDate: { $gte: new Date() } },
            attributes: ['id'],
            include: [includeTrial],
            order: [['id']],
            transaction
        });

        const certsIds = _.map(certs, 'id');
        const trialsByUid = _(certs)
            .groupBy('trial.user.uid')
            .mapValues(userCerts => _.map(userCerts, cert => cert.trial))
            .value();

        const usersData = _(trialsByUid)
            .keys()
            .map(uid => ({
                uid: parseInt(uid, 10),
                services: this._getServicesByTrials(trialsByUid[uid])
            }))
            .value();

        return { certsIds, usersData };
    }

    static *getNullifiedCertificates(uid) {
        const includeService = {
            model: Service,
            attributes: ['code']
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['slug', 'id'],
            include: [includeService]
        };
        const includeAuthType = {
            model: AuthType,
            attributes: [],
            where: { code: 'web' },
            as: 'authType'
        };
        const includeUser = {
            model: User,
            attributes: ['uid'],
            where: { uid },
            include: includeAuthType
        };
        const includeTrial = {
            model: Trial,
            where: { nullified: 0 },
            attributes: ['trialTemplateId'],
            include: [includeUser, includeTrialTemplate]
        };
        const includeType = {
            model: Type,
            where: { code: 'cert' },
            attributes: []
        };
        const certAttributes = ['id', 'confirmedDate', 'dueDate'];

        const certs = yield Certificate.findAll({
            where: { active: 0, dueDate: { $gte: new Date() } },
            attributes: certAttributes,
            include: [includeTrial, includeType],
            order: [['confirmedDate', 'DESC']]
        });

        return certs.map(certData => {
            const cert = _.pick(certData, certAttributes);

            this._setExamInfo(certData, cert);

            cert.service = _.get(certData, 'trial.trialTemplate.service.code');

            return cert;
        });
    }

    static *getCertificatesInfo(certIds) {
        const includeUser = {
            model: User,
            attributes: ['login']
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['slug']
        };
        const includeTrial = {
            model: Trial,
            attributes: ['id', 'nullified'],
            include: [includeUser, includeTrialTemplate]
        };
        const certs = yield Certificate.findAll({
            where: { id: certIds },
            attributes: ['id', 'active', 'confirmedDate', 'dueDate'],
            include: includeTrial,
            order: [['id']]
        });

        return certs.map(cert => ({
            id: cert.id,
            isActive: Boolean(cert.active),
            confirmedDate: cert.confirmedDate,
            dueDate: cert.dueDate,
            trialId: _.get(cert, 'trial.id'),
            isTrialNullified: Boolean(_.get(cert, 'trial.nullified')),
            login: _.get(cert, 'trial.user.login'),
            examSlug: _.get(cert, 'trial.trialTemplate.slug')
        }));
    }

    static *sendCertificatesToGeoadv(certsData, transaction) {
        const { host, path } = config.geoadv;
        const url = `${host}${path}`;
        const processedCertIds = [];

        for (const certData of certsData) {
            try {
                const tvmTicket = yield getTvmTicket(config.tvm.geoadv);
                const options = {
                    method: 'PUT',
                    headers: {
                        'X-Ya-Service-Ticket': tvmTicket,
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(certData)
                };

                yield got(url, options);

                processedCertIds.push(certData.certId);
            } catch (error) {
                log.error('Request to geoadv was failed', { error, certData });

                yield Notifier.failedRequestToGeoadv({ error, certData });
            }
        }

        if (!_.isEmpty(processedCertIds)) {
            yield Certificate.update(
                { isSentToGeoadv: true },
                {
                    where: { id: processedCertIds },
                    fields: ['isSentToGeoadv'],
                    transaction
                }
            );
        }
    }

    static *getGeoadvCertificates() {
        const includeUser = {
            model: User,
            attributes: ['uid']
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            where: { slug: config.geoadv.exams },
            attributes: ['slug']
        };
        const includeTrial = {
            model: Trial,
            where: { nullified: 0 },
            attributes: ['id'],
            include: [includeTrialTemplate, includeUser]
        };
        const certs = yield Certificate.findAll({
            where: {
                active: 1,
                isSentToGeoadv: false
            },
            attributes: ['id'],
            include: includeTrial
        });

        return certs.map(cert => ({
            certId: cert.id,
            uid: _.get(cert, 'trial.user.uid'),
            examSlug: _.get(cert, 'trial.trialTemplate.slug')
        }));
    }
}

module.exports = CertificateModel;
