'use strict';

const _ = require('lodash');
const got = require('got');
const config = require('yandex-config');
const log = require('logger');
const moment = require('moment');
const sleep = require('helpers/sleep');
const mailer = require('helpers/mailer');
const assert = require('helpers/assert');
const fs = require('fs');
const AgenciesLoader = require('models/agenciesLoader');
const Excel = require('models/excel');
const WebUser = require('models/user/webUser');
const getTvmTicket = require('helpers/tvm').getTicket;

const {
    Trial,
    Certificate,
    User,
    Agency,
    Role,
    DirectSync,
    AuthType
} = require('db/postgres');

class Direct {
    static *selectUIDs() {
        const now = new Date();
        const includeCertificates = {
            model: Certificate,
            as: 'certificates',
            attributes: ['dueDate'],
            where: { dueDate: { $gt: now } }
        };
        const includeTrials = {
            model: Trial,
            as: 'trials',
            attributes: ['id', 'userId'],
            include: includeCertificates,
            where: { nullified: 0 }
        };
        const includeAuthType = {
            model: AuthType,
            where: { code: 'web' },
            attributes: ['id'],
            as: 'authType'
        };

        const users = yield User.findAll({
            attributes: ['uid'],
            include: [includeTrials, includeAuthType]
        });

        return users.map(user => user.get('uid'));
    }

    static *getAgencyInfo(uid) {
        const tvmTicket = yield getTvmTicket(config.tvm.direct);
        const options = {
            query: { uid, json: 1 },
            headers: {
                'x-ya-service-ticket': tvmTicket,
                Connection: 'keep-alive'
            },
            json: true
        };

        const { body } = yield got(config.direct.url, options);

        return {
            agency: {
                login: _.get(body, 'chief_agency_login'),
                title: _.get(body, 'chief_agency_login'),
                manager: _.get(body, 'manager_login'),
                directId: _.get(body, 'client_id')
            },
            role: _.get(body, 'role'),
            uid
        };
    }

    static *makeRequestsToDirect(uids) {
        const agenciesData = [];

        for (let i = 0; i < uids.length; i += 1) {
            yield sleep(config.direct.sleep);

            try {
                const agencyInfo = yield Direct.getAgencyInfo(uids[i]);

                agenciesData.push(agencyInfo);
            } catch (err) {
                log.warn('syncAgenciesInfoFailed makeRequestsToDirect', { uid: uids[i], err });
            }
        }

        return agenciesData;
    }

    static *_getRoleId(data) {
        const role = _.get(data, 'role', 'user');
        const roleData = yield Role.findOrCreate({
            attributes: ['id'],
            where: { code: role },
            defaults: {
                code: role,
                title: role,
                active: 1
            }
        });

        return roleData[0].get('id');
    }

    static *_getAgencyId(data) {
        const agencyInstance = yield Agency.findOrCreate({
            attributes: ['id'],
            where: { login: data.agency.login },
            defaults: data.agency
        });
        const [, isCreated] = agencyInstance;

        if (!isCreated) {
            yield Agency.update(data.agency, {
                fields: ['manager', 'directId'],
                where: { login: data.agency.login }
            });
        }

        return agencyInstance[0].get('id');
    }

    static *updateUserAndAgency(data) {
        const agencyLogin = _.get(data, 'agency.login');
        const user = yield WebUser.findUser({ where: { uid: data.uid } });

        if (!agencyLogin) {
            log.warn('User does not belong to any agencies', { uid: data.uid });

            user.set('agencyId', null);
            yield user.save();

            return false;
        }

        const agencyId = yield Direct._getAgencyId(data);
        const roleId = yield Direct._getRoleId(data);

        user.set('agencyId', agencyId);
        user.set('roleId', roleId);
        yield user.save();

        log.info('User.agencyId has been updated', { uid: data.uid });

        return true;
    }

    static *_getAgenciesExcelFile(startTime) {
        const data = yield AgenciesLoader.getFlatData(startTime);

        const blank = fs.readFileSync('templates/agenciesBlank.xlsx');
        const excel = Excel.tryLoad(blank);

        return excel.write(data, true);
    }

    static *_checkSyncOnAvailability() {
        const records = yield DirectSync.findAll({
            order: [['startTime', 'DESC']],
            limit: 1
        });

        const [record] = records;

        if (!record) {
            return;
        }

        const startTime = record.get('startTime');
        const syncDelay = config.direct.delay;
        const isLastSyncExpired = new Date() - new Date(startTime) > syncDelay;
        const finishTime = Boolean(record.get('finishTime'));
        const isAvailable = finishTime || isLastSyncExpired;

        assert(isAvailable, 403, 'Sync has already begun', 'SAB', {
            availabilityTime: moment(startTime).add(syncDelay).startOf('minute').toISOString()
        });
    }

    static *_sendFile(record) {
        const startTime = record.get('startTime');
        const now = moment(startTime).locale('ru').format('lll');
        const options = {
            from: _.get(config, 'mailLists.agencies.from'),
            to: _.get(config, 'mailLists.agencies.to'),
            subject: `Данные сертификатов по агентствам (${now})`,
            text: ''
        };

        try {
            const attachmentContent = yield Direct._getAgenciesExcelFile(startTime);
            const attachment = {
                filename: `${now}.xlsx`,
                content: attachmentContent
            };

            _.assign(options, { attachments: [attachment] });
        } catch (err) {
            log.warn('syncAgenciesInfoFailed _getAgenciesExcelFile', { err });

            const textData = _.assign(record.toJSON(), { err });

            _.assign(options, {
                to: _.get(config, 'mailLists.debug'),
                subject: `Ошибка при формировании файла статистики (${now})`,
                text: JSON.stringify(textData)
            });
        }

        yield mailer(options);

        log.info('Message has been successfully sent');
    }

    static *syncAgenciesInfo() {
        yield Direct._checkSyncOnAvailability();

        const record = yield DirectSync.create({
            startTime: new Date()
        });

        const uids = yield Direct.selectUIDs();
        const responseData = yield Direct.makeRequestsToDirect(uids);
        let updatedRowsCount = 0;

        for (let i = 0; i < responseData.length; i += 1) {
            try {
                const isSuccess = yield Direct.updateUserAndAgency(responseData[i]);

                updatedRowsCount += isSuccess;
            } catch (err) {
                log.warn('syncAgenciesInfoFailed updateUserAndAgency', { uid: responseData[i].uid, err });
            }

        }

        log.info('The relations between users and agencies have been updated', {
            allRowsCount: uids.length,
            updatedRowsCount
        });

        record.set('finishTime', new Date());
        yield record.save();

        yield Direct._sendFile(record);
    }

    static *getLastSyncDate() {
        const record = yield DirectSync.findOne({
            attributes: ['startTime'],
            where: { finishTime: { $ne: null } },
            order: [['startTime', 'DESC']]
        });

        if (!record) {
            return;
        }

        return record.get('startTime');
    }
}

module.exports = Direct;
