'use strict';

const _ = require('lodash');
const { createCanvas } = require('canvas');
const config = require('yandex-config');

const OPTIONS = config.draw;
const I18N = require('tanker/i18n.cert.json');

class DrawBase {

    /**
     * Базовый конструктор. Эта модель занимается рисованием сертификатов в форматах png и pdf
     * @param {object} data - данные для рисования сертификата
     * @param {string} format - формат (png|pdf)
     */
    constructor(data, format) {
        this._data = _.assign(data, {
            firstname: _.defaultTo(data.firstname, ' '),
            lastname: _.defaultTo(data.lastname, ' ')
        });
        this._format = format;
    }

    get russianService() {
        return ['direct_base', 'direct_pro', 'metrika', 'cpm', 'zen', 'rsya', 'market'];
    }

    get isRussianService() {
        return this.russianService.indexOf(this._data.service) !== -1;
    }

    get isCertificate() {
        return this._data.type === 'cert';
    }

    get language() {
        return this._data.language;
    }

    get name() {
        return [this._data.firstname, this._data.lastname]
            .filter(word => word.length)
            .join(' ');
    }

    get offset() {
        this._offset = this._offset || {
            left: this._getSize('padding') + this._getSize('borderLeft'),
            top: this._getSize('padding') + this._getSize('borderTop')
        };

        return this._offset;
    }

    /**
     * На основании текущего формата возвращает размер определенного элемента из конфига (OPTIONS.sizes)
     * @param {string} name - название параметра
     * @returns {number}
     * @private
     */
    _getSize(name) {
        return OPTIONS.sizes[this.type][this._format][name];
    }

    /**
     * Умножает переданную величину на множитель данного типа сертификата (для png - 1)
     * @param {number} value
     * @returns {number}
     * @private
     */
    _scaled(value) {
        return Math.round(value * this._getSize('scale'));
    }

    /**
     * Возвращает перевод ключа из файла tanker/i18n.cert.json
     * @param {string} key
     * @returns {string}
     * @private
     */
    _i18n(key) {
        const tankerLanguage = config.i18n.tankerLanguages[this.language];

        return _.get(I18N, [tankerLanguage, 'cert', key], '');
    }

    /**
     * Шрифт для рисования имени
     * @param {number} size
     * @returns {string}
     * @private
     */
    _nameFont(size) {
        return `${size}px "${OPTIONS.fonts.initials[this.locale]}"`;
    }

    /**
     * Шрифт для рисования всего, кроме инициалов и имени
     * @param {number} size
     * @returns {string}
     * @private
     */
    _textFont(size) {
        return `${size}px "${OPTIONS.fonts.text[this.language]}"`;
    }

    /**
     * Вычисление размеров шрифта для рисования имени на сертификате
     * @param {Number} fontSize - размер шрифта
     * @param {Number} width - ширина, доступная для слова
     * @returns {number}
     * @private
     */
    _getSizeForName(fontSize, width) {
        const { firstname, lastname } = this._data;

        const longestWord = firstname.length > lastname.length ? firstname : lastname;
        let size = this._scaled(fontSize);

        this._ctx.font = this._nameFont(size);

        while (this._ctx.measureText(longestWord).width > width) {
            size -= 1;
            this._ctx.font = this._nameFont(size);
        }

        return size;
    }

    /**
     * Рисование имени пользователя
     * @param {Object} options - объект опций
     * @param {Number} options.fontSize - начальный размер шрифта
     * @param {Number} options.width - ширина, доступная для слова
     * @param {Number} options.maxSizeForText - максимальная ширина надписи (имя + фамилия)
     * @param {Number} options.marginLeft - отступ слева
     * @param {Number} options.marginTop - отступ сверху
     * @param {Number} options.lineHeight - высота строки
     * @returns {Number} - нижняя граница написанного имени на холсте
     * @private
     */
    _drawName(options) {
        const size = this._getSizeForName(options.fontSize, options.width);
        let lineHeight = size;

        if (options.lineHeight) {
            const fontSizeDelta = options.fontSize - size;

            lineHeight = options.lineHeight - fontSizeDelta;
            options.marginTop = options.marginTop - fontSizeDelta;
        }

        this._ctx.fillStyle = '#000';
        this._ctx.font = this._nameFont(size);

        if (this._ctx.measureText(options.text).width > options.maxSizeForText) {
            this._ctx.fillText(this._data.firstname, options.marginLeft, options.marginTop);
            this._ctx.fillText(this._data.lastname, options.marginLeft, options.marginTop + lineHeight);

            return options.marginTop + (2 * lineHeight);
        }
        this._ctx.fillText(options.text, options.marginLeft, options.marginTop);

        return options.marginTop + lineHeight;

    }

    _drawStatus(options) {
        this._ctx.font = this._textFont(this._scaled(options.fontSize));
        this._ctx.fillText(this._i18n('status'), options.marginLeft, options.marginTop);
    }

    _createCanvas(height) {
        const width = this._getSize('width') + (2 * this._getSize('borderLeft'));
        const canvas = createCanvas(width, height, this._format);

        this._ctx = canvas.getContext('2d');
        this._ctx.fillStyle = '#fff';
        this._ctx.fillRect(0, 0, width, height);

        return canvas;
    }
}

module.exports = DrawBase;
