'use strict';

const _ = require('lodash');
const { Image } = require('canvas');
const moment = require('moment');
const OPTIONS = require('yandex-config').draw;

const DrawBase = require('models/draw/base');

class InitialsBase extends DrawBase {
    get type() {
        return 'initials';
    }

    get isInTheTimeInterval() {
        return new Date(this._data.confirmedDate) > new Date(2016, 1, 15);
    }

    get logo() {
        return _.get(OPTIONS, ['services', this.type, this._data.service, 'logo', this._format]);
    }

    get background() {
        return _.get(OPTIONS, ['services', this.type, this._data.service, 'background', this._format]);
    }
    get russianService() {
        return ['direct_base', 'direct_pro', 'metrika', 'cpm', 'zen', 'rsya', 'market'];
    }
    get isRussianService() {
        return this.russianService.indexOf(this._data.service) !== -1;
    }

    /**
     * Имеет ли пользователь в имени и фамилии иероглифы
     * @returns {boolean}
     */
    get hasCjkSymbols() {
        return new RegExp(OPTIONS.regex.cjkSymbol, 'gi').test(this.name);
    }

    /**
     * Возвращает левую границу расположения блока инициалов на сертификате
     * @returns {number}
     */
    get initialsBorder() {
        return this._getSize('width') * OPTIONS.initialsRatio[this.locale];
    }

    /**
     * Функция определяет принадлежность буквы той или иной группе специфичных символов
     * @param {string} letter - буква
     * @param {string} list - идентификатор группы символов в конфиге (OPTIONS.letters)
     * @returns {boolean}
     * @private
     */
    _isBelongsTo(letter, list) {
        return _.includes(OPTIONS.letters[list][this.locale], letter);
    }

    /**
     * Рисование логотипа в левом верхнем углу сертификата
     * @private
     */
    _drawLogo() {
        const img = new Image();

        img.src = this.logo;
        this._ctx.drawImage(img, this.offset.left, this.offset.top);
    }

    /*
     Рисование прямоугольника со скругленными углами
     */
    // eslint-disable-next-line max-params
    _roundedRect(x, y, width, height, radius, color) {
        this._ctx.fillStyle = color;
        this._ctx.beginPath();
        this._ctx.moveTo(x + radius, y);
        this._ctx.lineTo(x + width - radius, y);
        this._ctx.quadraticCurveTo(x + width, y, x + width, y + radius);
        this._ctx.lineTo(x + width, y + height - radius);
        this._ctx.quadraticCurveTo(x + width, y + height, x + width - radius, y + height);
        this._ctx.lineTo(x + radius, y + height);
        this._ctx.quadraticCurveTo(x, y + height, x, y + height - radius);
        this._ctx.lineTo(x, y + radius);
        this._ctx.quadraticCurveTo(x, y, x + radius, y);
        this._ctx.closePath();
        this._ctx.fill();
    }

    /**
     * Рисование номера сертификата в красном прямоугольнике
     * @param {number} certId
     * @private
     */
    _drawCertNumber(certId) {
        this._ctx.font = this._textFont(this._scaled(22));

        const certNumber = `${this._i18n('id')}${certId}`;
        const certNumberWidth = this._ctx.measureText(certNumber).width + this._scaled(20);
        const verticalOffset = this._scaled(375) + this.offset.top;

        this._roundedRect(this.offset.left, verticalOffset, certNumberWidth,
            this._scaled(36), this._scaled(6), '#DB1215');

        this._ctx.fillStyle = '#fff';
        this._ctx.fillText(certNumber, this.offset.left + this._scaled(10), this._scaled(400) + this.offset.top);
    }

    /**
     * Рисование всей информации, которая находится под именем пользователя
     * @param {string} dueDate - дата окончания срока действия сертификата
     * @private
     */
    _drawAdditionalInfo(dueDate) {
        this._ctx.fillStyle = '#000';

        const dueDateText = `${this._i18n('due')} ${moment(dueDate).format('DD.MM.YYYY')}`;
        const lineHeight = this._scaled(30);
        const top = this._getSize('height') + this._getSize('borderTop') - this._getSize('padding');

        this._ctx.font = this._textFont(this._scaled(22));
        this._ctx.fillText(dueDateText, this.offset.left, top - (2 * lineHeight));
        this._ctx.fillText(this._i18n('authenticity'), this.offset.left, top - lineHeight);

        const onSiteText = this._i18n('onsite');
        const text = `${onSiteText} `;
        const textWidth = onSiteText ? this._ctx.measureText(text).width : 0;

        this._ctx.fillText(text, this.offset.left, top);
        this._ctx.fillStyle = 'rgb(128, 130, 133)';
        this._ctx.fillText(this._i18n('link'), this.offset.left + textWidth, top);
    }

    /**
     * Рисование инициалов в правой части сертификата
     */
    _drawInitials() {
        const letters = this._getInitialsLetters();
        const translateOffsetX = this.initialsBorder + this._getSize('borderLeft');

        const img = new Image();

        img.src = this.background;

        this._ctx.translate(translateOffsetX, 0);
        this._ctx.fillStyle = this._ctx.createPattern(img, 'no-repeat');
        this._drawLetters(letters);
        this._ctx.translate(-translateOffsetX, 0);
    }

    /**
     * Входная точка для рисования сертификата
     * @returns {Buffer}
     */
    drawCert() {
        const certHeight = this._getSize('height') + (2 * this._getSize('borderTop'));
        const canvas = this._createCanvas(certHeight);

        this._drawLogo();

        const nameCoord = this._drawName({
            fontSize: 54,
            text: this._getFormattedName(),
            width: this.initialsBorder - this.offset.left,
            maxSizeForText: this.initialsBorder - (2 * this._getSize('borderLeft')),
            marginLeft: this.offset.left,
            marginTop: this._scaled(210) + this.offset.top
        });

        this._drawStatus({
            fontSize: 32,
            marginLeft: this.offset.left,
            marginTop: nameCoord
        });
        this._drawCertNumber(this._data.certId);
        this._drawAdditionalInfo(this._data.dueDate);

        this._drawInitials();

        return {
            buffer: canvas.toBuffer()
        };
    }
}

module.exports = InitialsBase;
