'use strict';

const _ = require('lodash');
const OPTIONS = require('yandex-config').draw;

const InitialsBase = require('models/draw/initials-base');

class CJK extends InitialsBase {
    get locale() {
        return 'cjk';
    }

    /*
    Эта модель применима только если:
    - получен после 15.02.2016
    - type === 'cert'
    - в имени есть иероглифы
    - имеет изображения бэкграунда и логотипа
     */
    get isSuitable() {
        return [
            this.isInTheTimeInterval,
            this.isCertificate,
            this.hasCjkSymbols,
            this.background,
            this.logo
        ].every(Boolean);
    }

    /*
    Получение форматированного имени пользователя. Для CJK - это фамилия + имя с вырезанными НЕиероглифами
     */
    _getFormattedName() {
        const name = `${this._data.lastname}${this._data.firstname}`;

        return name.replace(new RegExp(OPTIONS.regex.notCjkSymbol, 'gi'), '');
    }

    /**
     * Шрифт для рисования инициалов
     * @param {number} size
     * @returns {string}
     * @private
     */
    _initialsFont(size) {
        return `${size}px "${OPTIONS.fonts.initials.cjk}"`;
    }

    /*
    Для CJK в качестве инициалов выбирается один или два подряд идущих иероглифа фамилии
     */
    _getInitialsLetters() {
        const lastname = new RegExp(OPTIONS.regex.cjkSymbols, 'i').exec(this._data.lastname);

        return {
            single: _.get(lastname, '0', '')
        };
    }

    /**
     * Рисование переданных символов в правой части сертификата
     * @param {Object} letters - объект вида { single: '' }
     * @private
     */
    _drawLetters(letters) {
        let size = 500 * this._getSize('scale');
        let verticalShiftCoefficient = 0;
        const singleLetter = letters.single;

        if (singleLetter.length > 1) {
            size *= 0.65;
            verticalShiftCoefficient = 0.3;
        }

        this._ctx.font = this._initialsFont(size);
        this._ctx.textBaseline = 'bottom';

        const letterWidth = Math.floor(this._ctx.measureText(singleLetter).width);
        const letterHeight = Math.floor(this._ctx.measureText(singleLetter).emHeightAscent);
        const horizontalShift = Math.ceil(letterWidth * 0.3);
        const verticalShift = -Math.ceil(letterHeight * verticalShiftCoefficient);

        this._ctx.fillText(
            singleLetter,
            this.initialsBorder - letterWidth + horizontalShift,
            this._getSize('height') + verticalShift + this._getSize('borderTop')
        );
    }
}

module.exports = CJK;
