'use strict';

const OPTIONS = require('yandex-config').draw;

const InitialsBase = require('models/draw/initials-base');

class EUR extends InitialsBase {
    get locale() {
        return 'eur';
    }

    /*
     Эта модель применима только если:
     - получен после 15.02.2016
     - type === 'cert'
     - в имени нет иероглифов
     - имеет изображения бэкграунда и логотипа
     */
    get isSuitable() {
        return [
            this.isInTheTimeInterval,
            this.isCertificate,
            !this.hasCjkSymbols,
            this.background,
            this.logo
        ].every(Boolean);
    }

    _getFormattedName() {
        return this.name;
    }

    /**
     * Шрифт для рисования инициалов
     * @param {number} size
     * @returns {string}
     * @private
     */
    _initialsFont(size) {
        return `bold ${size}px "${OPTIONS.fonts.initials.eur}"`;
    }

    _getInitialsLetters() {
        return {
            first: this._data.firstname[0].toUpperCase(),
            last: this._data.lastname[0].toUpperCase()
        };
    }

    /**
     * Рисование переданных символов в правой части сертификата
     * @param {Object} letters - объект вида { first: '', last: '' }
     * @private
     */
    _drawLetters(letters) {
        const firstnameLetterSize = this._getSizeForLetter(letters, 840 * this._getSize('scale'));

        this._drawFirstnameLetter(letters.first, firstnameLetterSize);

        const lastnameLetterSize = Math.ceil(firstnameLetterSize * 0.7);

        this._drawLastnameLetter(letters.last, lastnameLetterSize);
    }

    /**
     * Вычисление размеров первой буквы инициалов
     * @param {Object} letters - объект, содержащий первую и вторую буквы
     * @param {number} size - исходный размер
     * @returns {number}
     * @private
     */
    // eslint-disable-next-line complexity
    _getSizeForLetter(letters, size) {
        if (this._isBelongsTo(letters.first, 'big')) {
            return size * 0.9;
        }

        let sumWidth = this._getSumLettersWidth(letters, size);

        let intersection = Math.floor(size * 0.15);

        if (this._isBelongsTo(letters.last, 'wide') || this._isBelongsTo(letters.first, 'wide')) {
            intersection = Math.floor(size * 0.3);
        }

        while (sumWidth > this.initialsBorder + intersection) {
            size -= 1;
            sumWidth = this._getSumLettersWidth(letters, size);
        }

        if (this._isBelongsTo(letters.last, 'tight')) {
            size *= 0.9;
        }

        return size;
    }

    _getSumLettersWidth(letters, size) {
        this._ctx.font = this._initialsFont(size);
        const firstLetterWidth = this._ctx.measureText(letters.first).width;

        this._ctx.font = this._initialsFont(Math.floor(size * 0.7));
        const lastLetterWidth = this._ctx.measureText(letters.last).width;

        return firstLetterWidth + lastLetterWidth;
    }

    _drawFirstnameLetter(letter, size) {
        this._ctx.textBaseline = 'hanging';
        let verticalShift = -Math.ceil(size * 0.2);

        if (this._isBelongsTo(letter, 'firstWithGlyph')) {
            this._ctx.textBaseline = 'top';
            verticalShift = -Math.floor(size * 0.05);
        }

        if (this._isBelongsTo(letter, 'wide')) {
            verticalShift -= Math.ceil(size * 0.15);
        }

        this._ctx.font = this._initialsFont(size);
        this._ctx.textAlign = 'left';
        this._ctx.fillText(letter, 0, verticalShift + this._getSize('borderTop'));
    }

    _drawLastnameLetter(letter, size) {
        this._ctx.font = this._initialsFont(size);
        this._ctx.textBaseline = 'alphabetic';

        let verticalShift = 0;
        const letterWidth = Math.floor(this._ctx.measureText(letter).width);
        let horizontalShift = Math.ceil(letterWidth * 0.1);

        if (this._isBelongsTo(letter, 'wide')) {
            horizontalShift = Math.ceil(letterWidth * 0.2);
        }

        if (this._isBelongsTo(letter, 'lastWithGlyph')) {
            this._ctx.textBaseline = 'bottom';
            verticalShift = Math.ceil(size * 0.1);
            horizontalShift = Math.ceil(letterWidth * 0.1);
        }

        const horizontalOffsetForLetter = this._getSize('width') - this.initialsBorder - letterWidth + horizontalShift;
        const verticalOffsetForLetter = this._getSize('height') + verticalShift + this._getSize('borderTop');

        this._ctx.fillText(letter, horizontalOffsetForLetter, verticalOffsetForLetter);
    }
}

module.exports = EUR;
