'use strict';
const { Image } = require('canvas');
const OPTIONS = require('yandex-config').draw;
const moment = require('moment');
const InitialsBase = require('models/draw/initials-base');
const TEXT = require('models/draw/data/old-eur.text.js');

class LevelCert extends InitialsBase {
    get locale() {
        return 'levelCert';
    }

    get type() {
        return 'levelCert';
    }

    get isSuitable() {
        return [
            this.isRussianService,
            this.isCertificate,
            this.background
        ].every(Boolean);
    }

    get scale() {
        return OPTIONS.sizes[this.type][this._format].scale;
    }

    _getText(name, param) {
        return TEXT[this.type][this._data.service][name](param);
    }

    _getFontCss(nameString, prop) {
        const scaleFields = ['letterSpacing', 'lineHeight', 'size', 'top', 'left'];

        let value = OPTIONS.fontsCss[this.type][this._data.service][nameString][prop];

        if (value === undefined) {
            value = OPTIONS.fontsCss[this.type][this._data.service][nameString][this._format][prop];
        } else {
            value = scaleFields.indexOf(prop) === -1 ? value : value * this.scale;
        }

        value = prop === 'size' ? `${value}px` : value;

        return value;
    }

    _drawOneLine(prop) {
        const letterSpacing = this._getFontCss(prop.nameString, 'letterSpacing');
        const bold = this._getFontCss(prop.nameString, 'bold');
        const size = this._getFontCss(prop.nameString, 'size');
        const fontFamily = this._getFontCss(prop.nameString, 'fontFamily');

        let left = this._getFontCss(prop.nameString, 'left');
        let letter;

        this._ctx.fillStyle = this._getFontCss(prop.nameString, 'color');
        this._ctx.font = `${bold} ${size} "${fontFamily}"`;
        for (let i = 0; i < prop.line.length; i += 1) {
            letter = prop.line[i];

            this._ctx.fillText(letter, left, prop.top);
            left += this._ctx.measureText(letter).width + letterSpacing;
        }
    }

    _drawLine(lines, nameString) {
        const lineHeight = this._getFontCss(nameString, 'lineHeight');
        let top = this._getFontCss(nameString, 'top');

        for (let i = 0; i < lines.length; i += 1) {
            this._drawOneLine({ line: lines[i], top, nameString });
            top += lineHeight;
        }
    }

    _drawCertNumber() {
        const certNumber = this._getText('certNumber', this._data.certId);

        this._drawLine(certNumber, 'certNumber');
    }

    _drawDue() {
        const due = this._getText('due');

        this._drawLine(due, 'due');
    }

    _drawDueDate() {
        const date_ = moment(this._data.dueDate).format('DD/ MM/ YYYY');
        const dueDate = this._getText('dueDate', date_);

        this._drawLine(dueDate, 'dueDate');
    }

    _drawName() {
        const fullName = [`${this._data.firstname} ${this._data.lastname}`];

        this._drawLine(fullName, 'name');
    }

    _drawDescription() {
        const description = this._getText('description');

        this._drawLine(description, 'description');
    }

    _drawLevel() {
        const level = this._getText('level');

        if (level) {
            this._drawLine(level, 'level');
        }
    }

    _drawNameCert() {
        const nameCert = this._getText('nameCert');

        this._drawLine(nameCert, 'nameCert');
    }

    _drawCheck() {
        const check = this._getText('check');

        this._drawLine(check, 'check');
    }

    _drawBackground() {
        const img = new Image();

        img.src = this.background;
        this._ctx.drawImage(img, 0, 0);
    }

    _drawQRcode() {
        const img = new Image();
        const _src = this._data.qrUrl;

        const left = this._getFontCss('qrCode', 'left');
        const top = this._getFontCss('qrCode', 'top');

        if (_src) {
            img.src = _src;
            this._ctx.drawImage(img, left, top);
        }
    }

    drawCert() {
        const certHeight = this._getSize('height');
        const canvas = this._createCanvas(certHeight);

        this._drawBackground();
        this._drawCertNumber();
        this._drawDue();
        this._drawDueDate();
        this._drawName();
        this._drawDescription();
        this._drawNameCert();
        this._drawCheck();
        this._drawLevel();
        this._drawQRcode();

        return {
            buffer: canvas.toBuffer()
        };
    }
}

module.exports = LevelCert;
