'use strict';

const _ = require('lodash');
const { Image } = require('canvas');
const moment = require('moment');
const OPTIONS = require('yandex-config').draw;

const DrawBase = require('models/draw/base');
const TEXT = require('models/draw/data/old-eur.text.js');

class OldEur extends DrawBase {
    get type() {
        return 'old';
    }

    get locale() {
        return 'eur';
    }

    get isInTheTimeInterval() {
        return new Date(this._data.confirmedDate) <= new Date(2016, 1, 15);
    }

    /*
     Эта модель применима только если:
     - получен до 15.02.2016
     - type === 'cert'
     - service === direct|metrika
     - существует background
     */
    get isSuitable() {
        return [
            this.isInTheTimeInterval,
            this.isCertificate,
            _.includes(['direct', 'metrika'], this._data.service),
            this.background
        ].every(Boolean);
    }

    get background() {
        return _.get(OPTIONS, ['services', this.type, this._data.service, 'background', this._format]);
    }

    get marginLeft() {
        return this.offset.left + this._scaled(100);
    }

    _getText(name) {
        return param => TEXT[this._data.service][name](param);
    }

    _multilineText(options) {
        this._ctx.font = options.font;
        options.lines.forEach((line, i) => {
            this._ctx.fillText(line, options.marginLeft, options.marginTop + (options.lineHeight * i));
        });
    }

    _drawBackground() {
        const img = new Image();

        img.src = this.background;
        this._ctx.drawImage(img, 0, 0);
    }

    _nameFont(size) {
        return this._textFont(size);
    }

    _drawAnnotation(options) {
        const annotation = this._getText('annotation')(this.name);

        this._multilineText({
            font: this._textFont(this._scaled(5)),
            lines: annotation,
            lineHeight: this._scaled(7),
            marginLeft: this.marginLeft,
            marginTop: options.marginTop
        });
    }

    _drawCertNumber() {
        const certNumber = this._getText('certNumber')(this._data.certId);

        this._multilineText({
            font: this._textFont(this._scaled(8)),
            lines: certNumber,
            lineHeight: this._scaled(10),
            marginLeft: this.offset.left,
            marginTop: this.offset.top
        });
    }

    _drawAdditionalInfo() {
        const marginTop = this.offset.top + this._scaled(250);
        const lineHeight = this._scaled(7);

        this._ctx.font = this._textFont(this._scaled(5));

        const dueDateText = `${this._i18n('due')} ${moment(this._data.dueDate).format('DD.MM.YYYY')}`;

        this._ctx.fillText(dueDateText, this.offset.left, marginTop + lineHeight);

        const text = `${this._i18n('onsite')} `;
        const textWidth = this._ctx.measureText(text).width;

        this._ctx.fillText(this._i18n('authenticity'), this.marginLeft, marginTop);
        this._ctx.fillText(text, this.marginLeft, marginTop + lineHeight);
        this._ctx.fillStyle = 'rgb(128, 130, 133)';
        this._ctx.fillText(this._i18n('link'), this.marginLeft + textWidth, marginTop + lineHeight);
    }

    /**
     * Входная точка для рисования сертификата
     * @returns {Buffer}
     */
    drawCert() {
        const certHeight = this._getSize('height');
        const canvas = this._createCanvas(certHeight);

        this._drawBackground();

        const layoutWidth = this._getSize('width') - this.marginLeft - this.offset.left;
        const nameCoord = this._drawName({
            fontSize: 12,
            text: this.name,
            width: layoutWidth,
            maxSizeForText: layoutWidth,
            marginLeft: this.marginLeft,
            marginTop: this.offset.top
        });

        this._drawStatus({
            fontSize: 8,
            marginLeft: this.marginLeft,
            marginTop: nameCoord + this._scaled(7)
        });

        this._drawAnnotation({
            marginTop: nameCoord + this._scaled(23)
        });
        this._drawCertNumber();
        this._drawAdditionalInfo();

        return {
            buffer: canvas.toBuffer()
        };
    }
}

module.exports = OldEur;
