'use strict';

const OPTIONS = require('yandex-config').draw;

const { createCanvas, Image } = require('canvas');
const _ = require('lodash');

const DrawBase = require('models/draw/base');

class YaLetterBase extends DrawBase {
    get type() {
        return 'yaLetter';
    }

    /*
     Эта модель применима только если:
     - получен после 18.08.2017
     - type === 'cert'
     - совпадает формат ('pdf или 'png')
     - имеет изображения бэкграунда и логотипа
     - не для русских сертификатов
     */
    get isSuitable() {
        return [
            this.isInTheTimeInterval,
            this.isCertificate,
            this.isCorrectFormat,
            this.background,
            !this.isRussianService
        ].every(Boolean);
    }

    get isCorrectFormat() {
        return this.format === this._format;
    }

    get isInTheTimeInterval() {
        return new Date(this._data.confirmedDate) > new Date(2017, 7, 18);
    }

    get background() {
        return _.get(OPTIONS, ['services', this.type, this.language, 'background', this._format]);
    }

    get proctoringLabel() {
        return _.get(OPTIONS, ['services', this.type, this.language, 'proctoringLabel', this._format]);
    }

    get logo() {
        return _.get(OPTIONS, ['services', this.type, this.language, 'logo', this._format]);
    }

    get service() {
        return _.get(OPTIONS, ['tankerKeys', this._data.service], '');
    }

    _createCanvas() {
        const width = this._getSize('width');
        const height = this._getSize('height');
        const canvas = createCanvas(width, height, this._format);

        this._ctx = canvas.getContext('2d');

        return canvas;
    }

    _drawStatus(options) {
        const lineHeight = this._getSize('lineHeight').status;

        this._ctx.fillStyle = '#000';
        this._ctx.font = this._textFont(options.fontSize);

        let firstLine = this._i18n('specialist');
        let secondLine = this._i18n(this.service);

        if (this.language !== 'ru') {
            firstLine = this._i18n(this.service);
            secondLine = this._i18n('specialist');
        }

        this._ctx.fillText(firstLine, options.marginLeft, options.marginTop - lineHeight);
        this._ctx.fillText(secondLine, options.marginLeft, options.marginTop);
    }

    _drawBackground() {
        const img = new Image();

        img.src = this.background;
        this._ctx.drawImage(img, 0, 0);
    }

    _drawProctoringMark(options) {
        if (!this._data.isProctoring) {
            return;
        }

        const img = new Image();

        img.src = this.proctoringLabel;
        this._ctx.drawImage(img, options.marginLeft, options.marginTop);
    }

    _scaled(value) {
        return value;
    }

    _nameFont(size) {
        return `${size}px "${OPTIONS.fonts.text[this.language]}"`;
    }

    _getFormattedName() {
        return this.name;
    }
}

module.exports = YaLetterBase;
