'use strict';

const {
    Answer,
    Category,
    Question,
    Section,
    Service,
    TrialTemplate,
    TrialTemplateAllowedFails,
    TrialTemplateToSection,
    Type
} = require('db/postgres');

const Base = require('models/base');

const assert = require('helpers/assert');
const _ = require('lodash');

/**
 * @see: https://github.yandex-team.ru/expert/expert/blob/a07b88bf60644c146b5989c2a7030167755cd5fd/src/main/java/ru/yandex/expert/model/Language.java#L13-L15
 */
const EXAM_FIELDS = [
    'id',
    'active',
    'title',
    'description',
    'rules',
    'seoDescription',
    'ogDescription',
    'previewCert',
    'allowedTriesCount',
    'timeLimit',
    'delays',
    'validityPeriod',
    'periodBeforeCertificateReset',
    'delayUntilTrialsReset',
    'slug',
    'language',
    'isProctoring',
    'clusterSlug'
];

const EXAM_CLUSTER_FIELDS = [
    'id',
    'active',
    'slug',
    'isProctoring',
    'clusterSlug'
];

const EXAM_UPDATED_FIELDS = [
    'title',
    'timeLimit',
    'description',
    'rules',
    'seoDescription',
    'ogDescription'
];

class Exam extends Base {
    static *findByIdentity(identity, transaction) {
        const condition = Exam.getFindCondition(identity);
        const exam = yield TrialTemplate.findOne({
            where: condition,
            include: [Type, Service],
            attributes: EXAM_FIELDS.concat(['type_id', 'service_id']),
            transaction
        });

        assert(exam, 404, 'Test not found', 'TNF');

        const data = yield {
            trialTemplateAllowedFails: TrialTemplateAllowedFails.findAll({
                where: { trialTemplateId: exam.id },
                attributes: ['allowedFails'],
                transaction
            }),
            trialTemplateToSection: TrialTemplateToSection.findAll({
                where: { trialTemplateId: exam.id, quantity: { $ne: 0 } },
                attributes: ['quantity', 'sectionId'],
                transaction
            })
        };

        exam.questionsCount = _.sumBy(data.trialTemplateToSection, 'quantity');
        exam.allowedFails = _.sumBy(data.trialTemplateAllowedFails, 'allowedFails');
        exam.sectionsCount = _.uniqBy(data.trialTemplateToSection, 'sectionId').length;

        return new Exam(exam);
    }

    static getFindCondition(identity) {
        if (/^\d+$/.test(identity)) {
            return { id: identity };
        }

        assert.isSlug(identity, 400, 'Exam identity is invalid', 'EII', { identity });

        return { slug: identity };
    }

    static *findByCluster(clusterSlug) {
        const exams = yield TrialTemplate.findAll({
            where: { clusterSlug },
            attributes: EXAM_CLUSTER_FIELDS,
            order: [['id']]
        });

        assert(!_.isEmpty(exams), 404, 'Exams by cluster not found', 'ECN', { clusterSlug });

        return exams;
    }

    static *getInfoByIds(trialTemplateIds) {
        const exams = yield TrialTemplate.findAll({
            where: { id: trialTemplateIds },
            attributes: ['id', 'slug', 'language']
        });

        return exams.reduce((result, exam) => {
            result[exam.id] = { slug: exam.slug, language: exam.language };

            return result;
        }, {});
    }

    static *getExamData(trialTemplateId, transaction) {
        const {
            sectionsData,
            categories,
            questionsData,
            toSections
        } = yield Exam._selectSectionsData(trialTemplateId, transaction);

        const {
            sections,
            trialTemplateAllowedFails,
            sectionsById
        } = Exam._getTransformedSectionsData(sectionsData);

        const categoriesById = _.transform(categories, (res, item) => {
            res[item.id] = item;
        }, {});

        const {
            questions,
            answers
        } = Exam._getTransformedQuestionsData(questionsData, sectionsById, categoriesById);

        const trialTemplateToSections = _.map(toSections,
            ({ trialTemplateId: examDataTrialTemplateId, sectionId, categoryId, quantity }) => ({
                trialTemplateId: examDataTrialTemplateId,
                quantity,
                sectionId: sectionsById[sectionId],
                categoryId: categoriesById[categoryId]
            })
        );

        return {
            sections,
            categories,
            trialTemplateAllowedFails,
            trialTemplateToSections,
            questions,
            answers
        };
    }

    static *_selectSectionsData(trialTemplateId, transaction) {
        const toSections = yield TrialTemplateToSection.findAll({
            where: {
                trialTemplateId,
                quantity: { $gt: 0 }
            },
            attributes: ['trialTemplateId', 'sectionId', 'categoryId', 'quantity'],
            raw: true,
            order: [
                ['sectionId'],
                ['categoryId']
            ],
            transaction
        });

        const data = _.transform(toSections, (result, { sectionId, categoryId }) => {
            result.sectionIds.add(sectionId);
            result.categoryIds.add(categoryId);
            result.pairs.push({ sectionId, categoryId });
        }, {
            sectionIds: new Set(),
            categoryIds: new Set(),
            pairs: []
        });

        const trialTemplateAllowedFails = {
            model: TrialTemplateAllowedFails,
            where: { trialTemplateId },
            attributes: ['trialTemplateId', 'sectionId', 'allowedFails'],
            as: 'trialTemplateAllowedFails'
        };

        const sectionsCondition = {
            model: Section,
            where: { id: Array.from(data.sectionIds) },
            attributes: ['id', 'serviceId', 'code', 'title'],
            include: trialTemplateAllowedFails,
            order: [['id']],
            transaction
        };

        const categoriesCondition = {
            model: Category,
            where: { id: Array.from(data.categoryIds) },
            attributes: ['id', 'difficulty', 'timeLimit'],
            raw: true,
            order: [['id']],
            transaction
        };

        return yield {
            sectionsData: Section.findAll(sectionsCondition),
            categories: Category.findAll(categoriesCondition),
            questionsData: Exam._selectQuestions(data.pairs, transaction),
            toSections
        };
    }

    static *_selectQuestions(pairs, transaction) {
        const includeAnswers = {
            model: Answer,
            as: 'answers',
            attributes: ['id', 'correct', 'text', 'active'],
            where: { active: 1 }
        };

        return yield Question.findAll({
            attributes: ['id', 'active', 'sectionId', 'categoryId', 'text', 'type'],
            where: {
                active: 1,
                $or: pairs
            },
            include: includeAnswers,
            order: [
                ['sectionId'],
                ['categoryId'],
                ['id'],
                [includeAnswers, 'id']
            ],
            transaction
        });
    }

    static _getTransformedSectionsData(sectionsData) {
        return _.transform(sectionsData, (res, item) => {
            const section = _.pick(item, ['serviceId', 'code', 'title']);

            res.sections.push(section);
            res.trialTemplateAllowedFails.push(_.assign(
                { sectionId: section },
                _.pick(item.trialTemplateAllowedFails[0], ['trialTemplateId', 'allowedFails'])
            ));
            res.sectionsById[item.id] = section;
        }, {
            sections: [],
            trialTemplateAllowedFails: [],
            sectionsById: {}
        });
    }

    static _getTransformedQuestionsData(questionsData, sectionsById, categoriesById) {
        return _.transform(questionsData, (res, item) => {
            const question = _.assign(
                {
                    sectionId: sectionsById[item.sectionId],
                    categoryId: categoriesById[item.categoryId]
                },
                _.pick(item, ['id', 'active', 'text', 'type'])
            );
            const answers = _.map(item.answers, answer => _.assign(
                { questionId: question },
                _.pick(answer, ['id', 'correct', 'text', 'active'])
            ));

            res.questions.push(question);
            res.answers = res.answers.concat(answers);
        }, {
            answers: [],
            questions: []
        });
    }

    get fields() {
        return EXAM_FIELDS.concat([
            'service',
            'type',
            'questionsCount',
            'allowedFails',
            'sectionsCount'
        ]);
    }

    *updateSettings(settings, transaction) {
        EXAM_UPDATED_FIELDS.forEach(field => {
            if (!settings[field]) {
                return;
            }

            this.set(field, settings[field]);
        });

        yield this.save({ transaction });
    }
}

module.exports = Exam;
