'use strict';

const XLSX = require('xlsx');
const assert = require('helpers/assert');
const _ = require('lodash');
const AdminBase = require('models/adminBase');

const ALPHABET = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
const COLUMNS_CHECKS = {
    examId: _.isNumber,
    sectionsCode: _.isString,
    sectionTitle: _.isString,
    allowedFails: _.isNumber,
    categoryId: _.isNumber,
    quantity: _.isNumber,
    questionId: _.isNumber,
    questionActive: _.isNumber,
    questionText: _.isString,
    questionType: _.isString,
    answerId: _.isNumber,
    answerActive: _.isNumber,
    answerText: () => true,
    answerCorrect: _.isNumber
};

class Excel extends AdminBase {
    constructor(workbook) {
        super();
        const [firstSheetName] = workbook.SheetNames;

        this._worksheet = workbook.Sheets[firstSheetName];
        this._columns = {};
        this._workbook = workbook;

        ALPHABET
            .split('')
            .filter(letter => this.worksheet[`${letter}2`])
            .map(letter => this.worksheet[`${letter}2`].v)
            .forEach((name, i) => {
                this._columns[name] = ALPHABET[i];
            });
    }

    get worksheet() {
        return this._worksheet;
    }

    get columns() {
        return this._columns;
    }

    get workbook() {
        return this._workbook;
    }

    static _validateValueType(columnName, value, rowNumber) {
        const check = COLUMNS_CHECKS[columnName];

        assert(check(value), 400, 'Invalid value type', 'IVT', { columnName, value, rowNumber });
    }

    getValue(columnName, rowNumber) {
        assert(this.columns[columnName], 400, `Invalid column name`, 'ICN', { columnName });

        const cellAddress = this.columns[columnName] + rowNumber;
        const cellValue = _.get(this.worksheet, `${cellAddress}.v`);

        if (cellValue) {
            Excel._validateValueType(columnName, cellValue, rowNumber);
        }

        if (!_.isString(cellValue)) {
            return cellValue;
        }

        return _(cellValue)
            .replace(/\r|\n/g, ' ')
            .replace(/\s+/g, ' ')
            .trim();
    }

    setValue(columnName, rowNumber, value) {
        assert(this.columns[columnName], 400, `Invalid column name`, 'ICN', { columnName });

        const cellAddress = this.columns[columnName] + rowNumber;
        const cellData = { v: value, t: _.isNumber(value) ? 'n' : 's' };

        this.worksheet[cellAddress] = cellData;
    }

    static tryLoad(data) {
        let xlsx;

        try {
            xlsx = XLSX.read(data);
        } catch (e) {
            assert(false, 400, 'Parse failed', 'PFD', { details: e.message });
        }

        return new Excel(xlsx);
    }

    *getOperations() {
        const operations = {};
        let context = {};

        for (let rowNumber = 3; this.getValue('answerText', rowNumber); rowNumber += 1) {
            context = this._getRowData(rowNumber, context);
            yield this._getRowOperations(operations, context);
        }

        const uniqueOperations = this._dropDuplicates(operations);

        Excel._checkCorrectAnswers(uniqueOperations.answers);

        return uniqueOperations;
    }

    write(data, isFullFilling) {
        const context = {};

        data = Array.isArray(data) ? data : [data];
        data.forEach((item, i) => this._writeRow(item, i + 3, context, isFullFilling));

        const range = XLSX.utils.decode_range(this.worksheet['!ref']);

        range.e.r += data.length;
        this.worksheet['!ref'] = XLSX.utils.encode_range(range);

        return XLSX.write(this.workbook, { type: 'buffer', bookType: 'xlsx' });
    }

    // eslint-disable-next-line max-params
    _writeRow(item, rowNumber, context, isFullFilling) {
        let isPreviousColumnIdentical = true;

        for (const key in item) {
            if (item[key] === context[key] && isPreviousColumnIdentical && !isFullFilling) {
                continue;
            }

            isPreviousColumnIdentical = false;
            context[key] = item[key];
            this.setValue(key, rowNumber, item[key]);
        }
    }

    _getRowData(rowNumber, context) {
        return _(COLUMNS_CHECKS)
            .keys()
            .map(columnName => [
                columnName,
                _.defaultTo(this.getValue(columnName, rowNumber), context[columnName])
            ])
            .fromPairs()
            .value();
    }

    *_getRowOperations(operations, data) {
        for (let i = 0; i < this.schemas.length; i += 1) {
            const entity = yield this.schemas[i].getData(operations, data);
            const schemaName = this.schemas[i].name;

            operations[schemaName] = operations[schemaName] || [];
            operations[schemaName].push(entity);
        }
    }

    _dropDuplicates(operations) {
        this.schemas.forEach(schema => {
            const schemaName = schema.name;

            operations[schemaName] = _.uniqBy(operations[schemaName], item => {
                const uniqueValue = schema.getUniqueValue(item);

                return JSON.stringify(uniqueValue);
            });
        });

        return operations;
    }

    static _checkCorrectAnswers(answers) {
        _(answers)
            .filter(['active', 1])
            .groupBy('questionId.text')
            .mapValues(data => {
                const questionType = _.get(data, '0.questionId.type');
                const correctAnswers = _.sumBy(data, 'correct');
                const answersCount = data.length;
                const isOneAnswer = questionType === 0 && correctAnswers === 1;
                const isSeveralAnswers = questionType === 1 && (correctAnswers <= answersCount) && answersCount > 0;
                const isCorrect = isOneAnswer || isSeveralAnswers;

                assert(isCorrect, 400, 'Wrong number of correct answers', 'WCA', {
                    questionText: _.get(data, '0.questionId.text')
                });
            })
            .value();
    }
}

module.exports = Excel;
