'use strict';

const _ = require('lodash');
const config = require('yandex-config');
const log = require('logger');
const moment = require('moment');
const Sequelize = require('sequelize');

const { Freezing, Trial } = require('db/postgres');

class FreezingModel {
    static *freeze(uid, trialTemplateIds, transaction) {
        if (_.isEmpty(trialTemplateIds)) {
            return;
        }

        const startTime = new Date();
        const finishTime = moment(startTime).add(config.freezing.time, 'ms').toDate();

        const dataForCreate = trialTemplateIds.map(trialTemplateId => {
            return { frozenBy: uid, trialTemplateId, startTime, finishTime };
        });

        yield Freezing.bulkCreate(dataForCreate, { transaction });

        log.info('Service was frozen', { uid, finishTime, trialTemplateIds });

        return finishTime;
    }

    static *getLastAttemptFinish(trialTemplateIds) {
        const startDate = moment().subtract(24, 'hour').toDate();
        const trials = yield Trial.findAll({
            attributes: ['started', 'timeLimit'],
            where: {
                expired: 0,
                finished: null,
                started: { $gte: startDate },
                trialTemplateId: trialTemplateIds
            }
        });

        const lastFinishTime = _(trials)
            .map(trial => {
                const started = trial.get('started');
                const timeLimit = trial.get('timeLimit');

                return moment(started).add(timeLimit, 'ms').toDate().getTime();
            })
            .max();

        return moment(lastFinishTime).toISOString();
    }

    static *unfreeze(uid, trialTemplateIds, transaction) {
        const query = this._lastFrozenExamsQuery(trialTemplateIds, transaction);
        const frozenExams = yield Freezing.findAll(query);

        yield Freezing.update(
            {
                unfrozenBy: uid,
                finishTime: new Date()
            },
            {
                where: {
                    $or: frozenExams
                },
                fields: ['unfrozenBy', 'finishTime'],
                transaction
            }
        );

        log.info('Exams were unfrozen', { trialTemplateIds, uid });
    }

    static _lastFrozenExamsQuery(trialTemplateIds, transaction) {
        const query = {
            attributes: [
                'trialTemplateId',
                [Sequelize.fn('max', Sequelize.col('start_time')), 'startTime']
            ],
            group: ['trialTemplateId'],
            raw: true,
            transaction
        };

        if (trialTemplateIds) {
            query.where = { trialTemplateId: trialTemplateIds };
        }

        return query;
    }

    static *getCurrentFrozenExams() {
        const query = this._lastFrozenExamsQuery();

        query.where = { finishTime: { $gt: new Date() } };

        return yield Freezing.findAll(query);
    }
}

module.exports = FreezingModel;
