'use strict';

const _ = require('lodash');

const BaseModel = require('models/base');
const WebUserModel = require('models/user/webUser');
const {
    Admin,
    Ban,
    Certificate,
    GlobalUser,
    Trial,
    TrialTemplate,
    User
} = require('db/postgres');

class GlobalUserModel extends BaseModel {
    static *findById({ id, attributes }, transaction) {
        return yield GlobalUser.findById(id, { attributes, transaction });
    }

    static *getOrCreate(user, transaction) {
        if (user.globalUserId) {
            return user.globalUserId;
        }

        const globalUser = yield this._create(user.login, transaction);

        yield user.update({ globalUserId: globalUser.id }, { transaction });

        return globalUser.id;
    }

    static *_create(actualLogin, transaction) {
        return yield GlobalUser.create({
            actualLogin,
            isActive: true,
            isBanned: false
        }, { transaction });
    }

    static *chooseActualLogin(logins) {
        const [proLogin, certLogin] = yield [
            this._getLoginByFirstProctoring(logins),
            this._getLoginByLastCertificate(logins)
        ];

        return proLogin ? proLogin : certLogin;
    }

    static *_getLoginByFirstProctoring(logins) {
        const includeTrialTemplate = {
            model: TrialTemplate,
            where: { isProctoring: true },
            as: 'trialTemplate',
            attributes: []
        };
        const includeUsers = {
            model: User,
            where: { login: logins },
            attributes: ['login']
        };
        const [trial] = yield Trial.findAll({
            attributes: ['started'],
            include: [includeTrialTemplate, includeUsers],
            order: [['started']],
            limit: 1
        });

        return _.get(trial, 'user.login', '');
    }

    static *_getLoginByLastCertificate(logins) {
        const includeCertificates = {
            model: Certificate,
            as: 'certificates',
            attributes: ['active']
        };
        const includeUsers = {
            model: User,
            where: { login: logins },
            attributes: ['login']
        };
        const trials = yield Trial.findAll({
            attributes: ['started'],
            include: [includeUsers, includeCertificates],
            order: [['started', 'DESC']]
        });

        const trialWithCert = _.find(trials, trial => _.get(trial, 'certificates.0.active'));

        return trialWithCert ? _.get(trialWithCert, 'user.login') : _.get(trials, '0.user.login', logins[0]);
    }

    static *_mergeGlobalUsers(globalUserIds, transaction) {
        const globalUsers = yield this._getGlobalUsersWithBans(globalUserIds, transaction);
        const logins = _.map(globalUsers, 'actualLogin');
        const newActualLogin = yield this.chooseActualLogin(logins);
        const activeGlobalUser = _.find(globalUsers, { actualLogin: newActualLogin });
        const nonActiveGlobalIds = _.without(globalUserIds, activeGlobalUser.id);

        yield GlobalUser.update(
            { isActive: false },
            {
                where: { id: { $in: nonActiveGlobalIds } },
                transaction
            }
        );

        yield this._copyBans(globalUsers, activeGlobalUser, transaction);

        return activeGlobalUser.id;
    }

    static *_getGlobalUsersWithBans(globalUserIds, transaction) {
        const includeBans = {
            model: Ban,
            as: 'bans',
            attributes: [
                'adminId',
                'action',
                'startedDate',
                'expiredDate',
                'trialTemplateId',
                'reason',
                'userLogin'
            ]
        };

        return yield GlobalUser.findAll({
            where: { id: { $in: globalUserIds } },
            attributes: ['id', 'actualLogin', 'isBanned'],
            include: includeBans,
            order: [
                ['id'],
                [includeBans, 'startedDate']
            ],
            transaction
        });
    }

    static *_copyBans(globalUsers, activeGlobalUser, transaction) {
        const isSuperBanned = _.some(globalUsers, user => user.isBanned);

        if (isSuperBanned && !activeGlobalUser.isBanned) {
            yield this.updateSuperban({
                globalUserId: activeGlobalUser.id,
                isBanned: true
            }, transaction);
        }

        const actualGlobalUserId = activeGlobalUser.id;

        yield Ban.update(
            { isLast: false },
            {
                fields: ['isLast'],
                where: { globalUserId: actualGlobalUserId },
                transaction
            }
        );

        const dataForCopy = _(globalUsers)
            .filter(user => user.id !== actualGlobalUserId && !_.isEmpty(user.bans))
            .map(user => {
                return _.map(user.bans, ban => _.assign(
                    { globalUserId: actualGlobalUserId },
                    _.pick(ban, [
                        'adminId',
                        'trialTemplateId',
                        'action',
                        'startedDate',
                        'expiredDate',
                        'reason',
                        'userLogin'
                    ])
                ));
            })
            .flatten()
            .value();

        yield Ban.bulkCreate(dataForCopy, { transaction });
        yield this._setIsLast(actualGlobalUserId, isSuperBanned, transaction);
    }

    static *_setIsLast(activeGlobalUserId, isSuperBanned, transaction) {
        const activeGlobalUserBans = yield Ban.findAll({
            where: { globalUserId: activeGlobalUserId },
            attributes: [
                'id',
                'action',
                'startedDate',
                'expiredDate',
                'trialTemplateId'
            ],
            order: [['startedDate', 'DESC']],
            raw: true,
            transaction
        });

        const groupedByExams = _.groupBy(activeGlobalUserBans, 'trialTemplateId');
        const examIds = _.keys(groupedByExams);

        const banIdsToUpdate = _.map(examIds, examId => {
            const examBans = groupedByExams[examId];

            if (isSuperBanned) {
                const lastSuperBan = _.find(examBans, ban => _.isNull(ban.expiredDate) && ban.action === 'ban');

                return lastSuperBan.id;
            }

            return examBans[0].id;
        });

        yield Ban.update(
            { isLast: true },
            {
                fields: ['isLast'],
                where: { id: { $in: banIdsToUpdate } },
                transaction
            }
        );
    }

    static *associateUsers(users, transaction) {
        let globalUserId;

        let globalIds = _(users)
            .filter(user => user.globalUserId)
            .map(user => user.globalUserId)
            .uniq()
            .value();

        if (_.isEmpty(globalIds)) {
            const actualLogin = yield this.chooseActualLogin(_.map(users, 'login'));

            const globalUser = yield this._create(actualLogin, transaction);

            globalUserId = globalUser.id;
            globalIds = [globalUser.id];
        }

        if (globalIds.length === 1) {
            [globalUserId] = globalIds;
        }

        if (globalIds.length > 1) {
            globalUserId = yield this._mergeGlobalUsers(globalIds, transaction);
        }

        const allAssociatedUsers = yield WebUserModel.getAssociatedUsers(globalIds, transaction);
        const usersForUpdate = _(users)
            .concat(allAssociatedUsers)
            .uniqBy(user => user.id)
            .value();

        yield WebUserModel.setGlobalUserId({ users: usersForUpdate, globalUserId, transaction });
    }

    static *getBansInfo(globalUserId) {
        const includeAdmin = {
            model: Admin,
            attributes: ['login']
        };
        const includeBans = {
            model: Ban,
            attributes: [
                'startedDate',
                'expiredDate',
                'trialTemplateId',
                'reason',
                'userLogin',
                'action',
                'isLast'
            ],
            include: includeAdmin,
            as: 'bans'
        };

        return yield GlobalUser.findOne({
            where: { id: globalUserId, isActive: true },
            include: includeBans,
            attributes: ['actualLogin', 'isBanned'],
            order: [[includeBans, 'startedDate', 'DESC']]
        });
    }

    static *updateSuperban(banData, transaction) {
        const { globalUserId, isBanned } = banData;

        return yield GlobalUser.update(
            { isBanned },
            {
                where: { id: globalUserId },
                fields: ['isBanned'],
                transaction
            }
        );

    }
}

module.exports = GlobalUserModel;
