'use strict';

const { Role, Admin, AdminToRole } = require('db/postgres');

const Base = require('models/base');
const assert = require('helpers/assert');

class Idm extends Base {
    static *getActiveRoles() {
        const roles = yield Role.findAll({
            where: { active: 1 },
            attributes: ['code', 'title']
        });

        return roles.reduce((result, role) => {
            result[role.code] = role.title;

            return result;
        }, {});
    }

    static *getUsers() {
        const includeRole = {
            model: Role,
            where: { active: 1 },
            attributes: ['code']
        };
        const includeAdminToRole = {
            model: AdminToRole,
            as: 'adminToRoles',
            include: includeRole
        };
        const users = yield Admin.findAll({
            include: includeAdminToRole,
            attributes: ['login']
        });

        return users.map(user => {
            const login = user.get('login');
            const roles = user.get('adminToRoles').map(adminToRole => {
                return { expert: adminToRole.role.code };
            });

            return { login, roles };
        });
    }

    static *createRole(adminUser, code) {
        const role = yield this._findRole(code);
        const adminToRole = {
            adminId: adminUser.id,
            roleId: role.id
        };

        const response = yield AdminToRole.findOrCreate({
            where: adminToRole,
            defaults: adminToRole
        });
        const [, isCreated] = response;

        return isCreated ? {} : { warning: 'Role already exists' };
    }

    static *removeRole(login, code) {
        const admin = yield this._findAdmin(login);
        const role = yield this._findRole(code);
        const where = { adminId: admin.id, roleId: role.id };

        const removedCount = yield AdminToRole.destroy({ where });

        return removedCount ? {} : { warning: 'Role already removed' };
    }

    static *_findRole(code) {
        const role = yield Role.findOne({
            where: { code },
            attributes: ['id']
        });

        assert(role, 404, 'Role not found', 'RNF', { code });

        return role;
    }

    static *_findAdmin(login) {
        const admin = yield Admin.findOne({
            where: { login },
            attributes: ['id']
        });

        assert(admin, 404, 'Admin not found', 'ADF', { login });

        return admin;
    }
}

module.exports = Idm;
