'use strict';

const _ = require('lodash');

const log = require('logger');
const assert = require('helpers/assert');
const db = require('db/postgres');

const AdminBase = require('models/adminBase');

class Loader extends AdminBase {
    constructor(data) {
        super();
        this._data = data;
    }

    get data() {
        return this._data;
    }

    *upsert(transaction) {
        yield this._resetSectionsQuantity(transaction);

        for (let i = 0; i < this.schemas.length - 1; i += 1) {
            const schema = this.schemas[i];
            const items = yield this._upsertCollection(schema, transaction);

            this._replaceIdInRelations(this.data, schema, items, []);
        }

        yield this._upsertAnswers(transaction);
    }

    *select(trialTemplateId) {
        const data = yield this._selectData(trialTemplateId);

        return _(data)
            .filter(item => {
                const toSections = _.get(item, 'question.section.trialTemplateToSections');
                const categoryId = _.get(item, 'question.categoryId');

                return _.some(toSections, { categoryId });
            })
            .map(item => this._flatData(item.toJSON(), trialTemplateId))
            .value();
    }

    checkExamId(examId) {
        const trialTemplateId = _.get(this.data, 'trialTemplateAllowedFails[0].trialTemplateId');

        assert(examId === trialTemplateId, 400, 'Exam ids not match', 'ENM', { examId, trialTemplateId });
    }

    *_selectData(trialTemplateId) {
        const trialTemplateAllowedFails = {
            model: db.TrialTemplateAllowedFails,
            where: { trialTemplateId },
            attributes: ['allowedFails'],
            as: 'trialTemplateAllowedFails'
        };

        const trialTemplateToSections = {
            model: db.TrialTemplateToSection,
            where: { trialTemplateId },
            attributes: ['quantity', 'sectionId', 'categoryId'],
            as: 'trialTemplateToSections'
        };

        const includeSections = {
            model: db.Section,
            attributes: ['code', 'title'],
            include: [trialTemplateToSections, trialTemplateAllowedFails]
        };

        const includeQuestions = {
            model: db.Question,
            attributes: ['id', 'active', 'text', 'type', 'categoryId', 'version'],
            include: includeSections,
            where: { active: 1 }
        };

        return yield db.Answer.findAll({
            attributes: ['id', 'text', 'correct', 'active'],
            where: { active: 1 },
            include: includeQuestions,
            order: [
                'question.section.trialTemplateToSections.section_id',
                'question.section.trialTemplateToSections.category_id',
                'question.id',
                'question.version',
                'id'
            ],
            nest: true
        });
    }

    _flatData(data, trialTemplateId) {
        const result = {
            examId: () => parseInt(trialTemplateId, 10),
            sectionsCode: 'question.section.code',
            sectionTitle: 'question.section.title',
            allowedFails: 'question.section.trialTemplateAllowedFails.0.allowedFails',
            categoryId: 'question.categoryId',
            quantity: quantityData => {
                const items = _.get(quantityData, 'question.section.trialTemplateToSections');
                const categoryId = _.get(quantityData, 'question.categoryId');
                const item = _.find(items, { categoryId });

                return _.get(item, 'quantity');
            },
            questionId: 'question.id',
            questionActive: 'question.active',
            questionText: 'question.text',
            questionType: questionTypeData => _.get(questionTypeData, 'question.type') ? 'несколько' : 'один',
            answerId: 'id',
            answerActive: 'active',
            answerText: 'text',
            answerCorrect: 'correct'
        };

        for (const key in result) {
            if (!Object.prototype.hasOwnProperty.call(result, key)) {
                continue;
            }

            const value = result[key];

            result[key] = _.isString(value) ? _.get(data, value) : value(data);
        }

        return result;
    }

    *_upsertCollection(schema, transaction) {
        const result = [];
        const data = this.data[schema.name];

        assert(data, 400, 'Data is absent', 'DIA', { schema: schema.name });

        for (let i = 0; i < data.length; i += 1) {
            const storedData = yield schema.upsert.call(this, schema, data[i], transaction);

            result.push(storedData);
        }

        return result;
    }

    // eslint-disable-next-line complexity, max-params
    _replaceIdInRelations(data, storedSchema, storedEntities, path) {
        if (!storedSchema.key) {
            return;
        }

        for (const key in data) {
            if (!Object.prototype.hasOwnProperty.call(data, key)) {
                continue;
            }

            const value = data[key];
            const keyPath = path.concat(key);

            if (_.isArray(value) || _.isObject(value)) {
                this._replaceIdInRelations(value, storedSchema, storedEntities, keyPath);
            }

            if (key === storedSchema.key) {
                const condition = storedSchema.getUniqueValue(value);
                const storedEntity = _.find(storedEntities, condition);

                this._checkStoredEntity({ data, storedEntity, key, keyPath, condition });

                data[key] = storedEntity.id;
            }
        }
    }

    _checkStoredEntity(info) {
        if (!info.storedEntity) {
            log.error('DATA', JSON.stringify(info.data));
            log.error('STORED_SCHEMA.KEY', info.key);
            log.error('KEY_PATH', info.keyPath.join('.'));
            log.error('CONDITION', info.condition);
        }

        assert(info.storedEntity, 500, 'Not match the condition', 'NMC');
    }

    // eslint-disable-next-line max-statements, complexity
    *_upsertAnswers(transaction) {
        const data = this.data.answers;

        assert(data, 400, 'Answers is absent', 'AIA');

        const dataByQuestionId = _.groupBy(data, 'questionId');

        for (const questionId in dataByQuestionId) {
            if (!Object.prototype.hasOwnProperty.call(dataByQuestionId, questionId)) {
                continue;
            }

            const answers = dataByQuestionId[questionId];
            const questionData = this._updatedQuestions[questionId];

            questionData.id = questionId;

            const oldAnswersIds = this._getOldAnswersIds(answers);
            const storedAnswers = yield this._getStoredAnswers(oldAnswersIds, transaction);

            this._assertStoredAnswers(oldAnswersIds, storedAnswers);
            this._assertQuestionHasAnswers(answers, questionData);

            if (!questionData.active) {
                yield this._deactivateAnswers({ id: _.map(answers, 'id') }, transaction);
                continue;
            }

            const hasChangedAnswers = this._hasChangedAnswers(answers, storedAnswers);

            if (hasChangedAnswers || questionData.isVersionUpdated) {
                yield this._createAnswersWithNewVersion(answers, questionData, transaction);
            }
        }
    }

    *_getStoredAnswers(oldAnswersIds, transaction) {
        return yield db.Answer.findAll({
            order: 'id',
            where: { id: oldAnswersIds },
            attributes: ['id', 'text', 'correct'],
            raw: true,
            transaction
        });
    }

    _hasChangedAnswers(answers, storedAnswers) {
        const hasNewAnswers = _.some(answers, { id: 0 });
        const hasDeactivatedAnswers = _.some(answers, { active: 0 });
        const hasChangedAnswers = _.some(storedAnswers, storedAnswer => {
            const answer = _.find(answers, { id: storedAnswer.id });

            return answer.text !== storedAnswer.text || answer.correct !== storedAnswer.correct;
        });

        return hasNewAnswers || hasDeactivatedAnswers || hasChangedAnswers;
    }

    _assertStoredAnswers(oldAnswersIds, storedAnswers) {
        const storedIds = _.map(storedAnswers, 'id');
        const notFoundIds = _.difference(oldAnswersIds, storedIds);

        assert(_.isEmpty(notFoundIds), 404, 'Answers not found', 'ANF', { answerIds: notFoundIds });
    }

    _assertQuestionHasAnswers(answers, questionData) {
        const allAnswersDeactivated = _.every(answers, { active: 0 });
        const hasNoAnswers = questionData.active && allAnswersDeactivated;

        assert(!hasNoAnswers, 400, 'Question has no active answers', 'QNA', { questionId: questionData.id });
    }

    *_deactivateAnswers(condition, transaction) {
        yield db.Answer.update(
            { active: 0 },
            {
                where: condition,
                transaction
            }
        );
    }

    _getOldAnswersIds(answers) {
        return _(answers)
            .filter(answer => answer.id > 0)
            .map('id')
            .value();
    }

    *_upQuestionVersion(questionId, transaction) {
        const order = 'version DESC';
        const storedQuestion = yield db.Question.findById(questionId, { transaction, order });
        const { id, version, sectionId, categoryId, text, type } = storedQuestion;

        storedQuestion.set('active', 0);
        yield storedQuestion.save({ transaction });

        yield db.Question
            .build({
                id,
                sectionId,
                categoryId,
                text,
                type,
                active: 1,
                version: version + 1
            })
            .save({ transaction });
    }

    *_createAnswersWithNewVersion(answers, questionData, transaction) {
        const questionId = questionData.id;
        const oldAnswersIds = this._getOldAnswersIds(answers);
        const { isVersionUpdated, version } = questionData;
        const questionVersion = isVersionUpdated ? version : version + 1;

        yield this._deactivateAnswers({ id: { $in: oldAnswersIds } }, transaction);

        if (!isVersionUpdated) {
            yield this._upQuestionVersion(questionId, transaction);
        }

        const dataForCreate = _(answers)
            .filter('active')
            .map(answer => {
                const { questionId: answerQuestionId, correct, text, active } = answer;

                return {
                    questionId: answerQuestionId,
                    questionVersion,
                    correct,
                    text,
                    active
                };
            })
            .value();

        yield db.Answer.bulkCreate(dataForCreate, { transaction });
    }

    *_resetSectionsQuantity(transaction) {
        const trialTemplateId = _.get(this.data, 'trialTemplateToSections[0].trialTemplateId');

        if (!trialTemplateId) {
            return;
        }

        yield db.TrialTemplateToSection.update({ quantity: 0 }, {
            where: { trialTemplateId },
            fields: ['quantity'],
            transaction
        });
    }
}

module.exports = Loader;
