const _ = require('lodash');

const { Lock, Admin, TrialTemplate } = require('db/postgres');

class LockModel {
    static *lock(data, transaction) {
        const { trialTemplateId, adminId } = data;

        yield Lock.create({
            trialTemplateId,
            adminId,
            lockDate: new Date()
        }, { transaction });
    }

    static *unlock(rowId, transaction) {
        yield Lock.update(
            { unlockDate: new Date() },
            {
                where: { id: rowId },
                fields: ['unlockDate'],
                transaction
            }
        );
    }

    static *tryFindLast(trialTemplateId, transaction) {
        const includeAdmin = {
            model: Admin,
            attributes: ['login'],
            required: true
        };

        return yield Lock.findOne({
            where: { trialTemplateId },
            attributes: ['id', 'lockDate', 'unlockDate'],
            include: includeAdmin,
            order: [['lockDate', 'DESC']],
            transaction
        });
    }

    static *findLockedExams() {
        const includeAdmin = {
            model: Admin,
            attributes: ['login']
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['slug']
        };

        const locksData = yield Lock.findAll({
            where: { unlockDate: null },
            attributes: ['lockDate'],
            include: [includeAdmin, includeTrialTemplate]
        });

        return _.transform(locksData, (res, lock) => {
            const examSlug = _.get(lock, 'trialTemplate.slug');

            res[examSlug] = {
                lockDate: _.get(lock, 'lockDate'),
                login: _.get(lock, 'admin.login')
            };
        }, {});
    }
}

module.exports = LockModel;
