'use strict';

const _ = require('lodash');
const { format } = require('url');

const { avatarsService, mdsService } = require('yandex-config');
const request = require('request');
const log = require('logger');

const { parseString } = require('xml2js').Parser({ attrkey: 'attrs' });

// FIXME: можно убрать эту переменную:
//    вместо tryPutData сделать метод-ретрайер,
//    а эти методы перейдут в putToAvatars и putToMds
const REQUESTS = {
    mds: data => function (cb) {
        request.post({ url: data.url, headers: data.options, body: data.buffer }, cb);
    },
    avatars: data => function (cb) {
        request
            .post({ url: data.url, json: true }, cb)
            .form()
            .append('file', data.buffer, data.options);
    }
};

class Mds {
    static get _putMethod() {
        return {
            pdf: (filename, buffer) => this.putToMds(`${filename}.pdf`, buffer),
            png: (fileName, buffer) => this.putToAvatars(fileName, buffer)
        };
    }

    // eslint-disable-next-line complexity
    static *tryPutData(data, service, tryCount) {
        tryCount = tryCount || 0;

        let statusCode;

        try {
            const res = yield REQUESTS[service](data);

            statusCode = _.get(res, '0.statusCode');

            if (statusCode !== 200) {
                throw new Error(`Fail to put certificate to ${service}`);
            }

            return res;
        } catch (err) {
            if (tryCount <= data.maxRepeatCount && (!statusCode || statusCode >= 500)) {
                return yield this.tryPutData(data, service, tryCount + 1);
            }

            log.warn(`Fail to put certificate to ${service}`, { tryCount, err, maxRepeatCount: data.maxRepeatCount });
        }
    }

    static *putToAvatars(name, buffer) {
        const { maxRepeatCount, write, namespace } = avatarsService;

        const path = `/put-${namespace}/${name}`;
        const url = format(_.assign(
            { pathname: path },
            write
        ));
        const options = { filename: path, 'content-type': 'image/png' };
        const data = { url, buffer, options, maxRepeatCount };

        const res = yield this.tryPutData(data, 'avatars');

        if (!res) {
            return null;
        }

        return _(res)
            .get('0.body.sizes.orig.path', '')
            .split('/')
            .slice(2, 4)
            .join('/');
    }

    static *putToMds(name, buffer) {
        const { maxRepeatCount, write, namespace, token } = mdsService;

        const path = `/upload-${namespace}/${name}`;
        const url = format(_.assign(
            { pathname: path },
            write
        ));
        const options = {
            'content-length': Buffer.byteLength(buffer),
            authorization: `Basic ${token}`,
            'content-type': 'application/pdf'
        };
        const data = { url, buffer, options, maxRepeatCount };

        const res = yield this.tryPutData(data, 'mds');

        if (!res) {
            return null;
        }

        let result;

        try {
            result = yield parseString.bind(null, res[0].body);
        } catch (err) {
            log.warn('Fail to parse xml answer from MDS', err);

            return null;
        }

        return _.get(result, 'post.attrs.key', null);
    }

    static getAvatarsPath(imagePath) {
        if (!imagePath) {
            return '';
        }

        return `https://${avatarsService.read.hostname}/get-${avatarsService.namespace}/${imagePath}/orig`;
    }

    static getMdsPath(filePath) {
        if (!filePath) {
            return '';
        }

        return `http://${mdsService.read.hostname}/get-${mdsService.namespace}/${filePath}`;
    }

    static *putCert(buffer, certId, certFormat) {
        const timestamp = `${Date.now()}_`;
        const put = this._putMethod[certFormat];

        return yield put(`${timestamp}${certId}`, buffer);
    }
}

module.exports = Mds;
