'use strict';

const config = require('yandex-config');
const { sender } = config;

const _ = require('lodash');
const got = require('got');

const mailer = require('helpers/mailer');
const log = require('logger');

const proctoringTexts = {
    request: 'Запрос к api ProctorEdu закончился неудачей. Информация об ошибке:',
    response: 'Некорректный ответ от api ProctorEdu. Детали:'
};

function _buildExamUrl(language, slug) {
    const tld = config.tldByLang[language];

    return `${config.frontHostPrefix}${tld}${config.frontExamPath}${slug}`;
}

function _getExamsLinks(examsInfo) {
    return _.map(examsInfo, exam => {
        const { language, slug } = exam;
        const examUrl = _buildExamUrl(language, slug);

        return `<a href="${examUrl}">${slug}</a>`;
    });
}

class Notifier {
    static *freezeNotify(login, finishTime, examsInfo) {
        const examsLinks = _getExamsLinks(examsInfo);
        const frozenExams = examsLinks.join('<br>');

        const options = {
            from: _.get(config, 'freezing.from'),
            to: _.get(config, 'freezing.to'),
            subject: `Данные о заморозке сервиса`,
            html: `<p>Сервис ${config.frontHostPrefix}ru/adv/expert был закрыт для прохождения тестов:<br>
                   ${frozenExams}
                   <br><br>Доступ закрыл: ${login}, время окончания: ${finishTime}`
        };

        yield mailer(options);
    }

    static *unfreezeNotify(login, examsInfo) {
        const examsLinks = _getExamsLinks(examsInfo);
        const frozenExams = examsLinks.join('<br>');

        const options = {
            from: _.get(config, 'freezing.from'),
            to: _.get(config, 'freezing.to'),
            subject: `Данные о разморозке сервиса`,
            html: `Сервис ${config.frontHostPrefix}ru/adv/expert был открыт для прохождения тестов:<br>
                   ${frozenExams}
                   <br><br>Доступ открыл: ${login}`
        };

        yield mailer(options);
    }

    static *failedRequestNotify(data, type) {
        const options = {
            from: _.get(config, 'proctoring.fail.from'),
            to: _.get(config, 'proctoring.fail.to'),
            subject: `Проблемы с запросом к ProctorEdu`,
            html: `<p>${proctoringTexts[type]}</p>
                   <pre>${JSON.stringify(data, null, 4)}</pre>`
        };

        yield mailer(options);
    }

    static *failedS3Notify(data) {
        const options = {
            from: _.get(config, 's3.fail.from'),
            to: _.get(config, 's3.fail.to'),
            subject: 'Проблемы с загрузкой файлов в S3',
            html: `<p>Не удалось загрузить файл в S3</p>
                    <pre>${JSON.stringify(data, null, 4)}</pre>`
        };

        yield mailer(options);
    }

    static *sendLetterToUser(email, identity, data) {
        if (!email) {
            log.warn('Email is empty');

            return;
        }

        const options = {
            method: 'POST',
            path: `/api/0/sales/transactional/${sender.mailIds[identity]}/send`,
            query: { 'to_email': email },
            auth: sender.user,
            json: true
        };

        if (data) {
            const params = new URLSearchParams();

            params.append('args', JSON.stringify(data));

            options.body = params.toString();
            options.headers = {
                'Content-Type': 'application/x-www-form-urlencoded'
            };
        }

        try {
            yield got(sender.host, options);
        } catch (err) {
            log.error('Sending email was failed', { err });
        }
    }

    static *draftForModerationNotify(examSlug, username) {
        const options = {
            from: _.get(config, 'mailLists.draftForModeration.from'),
            to: _.get(config, 'mailLists.draftForModeration.to'),
            subject: `Черновик теста ${examSlug} отправлен на модерацию`,
            html: `<p>Логин редактора: ${username}</p>
                <p>Посмотреть черновик: <a>${config.officeUrl}/admin/${examSlug}?tab=questions</a></p>`
        };

        yield mailer(options);
    }

    static *failedCreateTicket(data) {
        const options = {
            from: _.get(config, 'tracker.fail.from'),
            to: _.get(config, 'tracker.fail.to'),
            subject: `Ошибка при создании тикета в ${data.queue}`,
            html: data.html
        };

        yield mailer(options);
    }

    static *failedFfmpegProcessing(data) {
        const options = {
            from: _.get(config, 'proctoring.fail.from'),
            to: _.get(config, 'proctoring.fail.to'),
            subject: 'Проблемы с конкатенацией файлов',
            html: `<p>Не удалось обработать видео:</p>
                <pre>${JSON.stringify(data, null, 4)}</pre>`
        };

        yield mailer(options);
    }

    static *failedUpdatingReport(errorMessage) {
        const options = {
            from: _.get(config, 'yt.fail.from'),
            to: _.get(config, 'yt.fail.to'),
            subject: 'Проблемы с загрузкой отчета в YT',
            html: `<p>Не удалось обновить данные в отчете. Текст ошибки: ${errorMessage}.</p>`
        };

        yield mailer(options);
    }

    static *failedRequestToGeoadv(errorData) {
        const options = {
            from: _.get(config, 'geoadv.fail.from'),
            to: _.get(config, 'geoadv.fail.to'),
            subject: 'Проблемы c запросом к geoadv',
            html: `<p>Информация об ошибке:</p>
                <pre>${JSON.stringify(errorData, null, 4)}</pre>`
        };

        yield mailer(options);
    }
}

module.exports = Notifier;
