'use strict';

const _ = require('lodash');
const yandexConfig = require('yandex-config');
const config = yandexConfig.proctoring;
const ffmpeg = require('fluent-ffmpeg');
const fs = require('fs');
const got = require('got');
const log = require('logger');
const { v4: uuidv4 } = require('uuid');

const getTvmTicket = require('helpers/tvm').getTicket;

const assert = require('helpers/assert');
const { removeFiles } = require('helpers/files');

const Notifier = require('models/notifier');
const S3 = require('models/s3');
const HttpsProxyAgent = require('helpers/httpsProxyAgent');

const MS_IN_MINUTE = 60 * 1000;
const INPUT_VIDEO_DIR = 'models/proctoring/input';
const OUTPUT_VIDEO_DIR = 'models/proctoring/output';

class ProctorEdu {
    // eslint-disable-next-line complexity
    static *_tryRequestToApi(options, requestName, tryCount) {
        tryCount = tryCount || 1;

        try {
            const res = yield got(config.host, options);

            return { statusCode: 200, body: res.body };
        } catch (err) {
            const statusCode = _.get(err, 'statusCode');

            if (tryCount <= config.maxRepeatCount && (!statusCode || statusCode >= 499)) {
                return yield this._tryRequestToApi(options, requestName, tryCount + 1);
            }

            if (statusCode !== 404 || requestName === 'session') {
                const message = _.get(err, 'message', '');

                yield Notifier.failedRequestNotify({ message, err }, 'request');
            }

            log.error('Request to ProctorEdu was failed', { err });

            return { statusCode };
        }
    }

    static *_request(options, requestName) {
        const tvmTicket = yield getTvmTicket(yandexConfig.tvm.gozora);

        const agent = new HttpsProxyAgent({
            rejectUnauthorized: false,
            keepAlive: true,
            keepAliveMsecs: 1000,
            maxSockets: 256,
            maxFreeSockets: 256,
            scheduling: 'lifo',
            proxy: 'http://go.zora.yandex.net:1080',
            headers: {
                'X-Ya-Client-Id': yandexConfig.clientId,
                'X-Ya-Service-Ticket': tvmTicket
            }
        });

        options = _.merge({
            method: 'GET',
            headers: {
                'X-Api-Key': config.apiKey
            },
            path: options.path,
            agent: {
                rejectUnauthorized: false,
                https: agent
            }
        }, options);

        return yield this._tryRequestToApi(options, requestName);
    }

    static *getSessionData(openId) {
        const options = {
            path: `${config.protocolPath}/${openId}`,
            headers: {
                'Content-Type': 'application/json'
            }
        };
        const { body } = yield this._request(options, 'session');

        if (_.isUndefined(body)) {
            return;
        }

        let sessionData = null;
        const errorData = { body, openId };

        try {
            sessionData = JSON.parse(body);
        } catch (err) {
            log.error('Can not parse body', { message: err.message, body });

            _.set(errorData, 'parseError', err.message);
        }

        if (!_.isEmpty(sessionData)) {
            return sessionData;
        }

        yield Notifier.failedRequestNotify(errorData, 'response');
    }

    static *getVideosByOpenId(openId) {
        const options = {
            path: `${config.eventsPath}/${openId}`,
            headers: {
                'Content-Type': 'application/json'
            }
        };

        const { body, statusCode } = yield this._request(options, 'videos');

        if (statusCode === 404) {
            return [];
        }

        let eventsData = null;

        try {
            eventsData = JSON.parse(body);
        } catch (err) {
            log.error('Can not parse body', { err, body });
        }

        if (eventsData) {
            return this._getVideosFromEvents(eventsData);
        }
    }

    static _getVideosFromEvents(events) {
        const sortedEvents = events.sort((event1, event2) => {
            return new Date(event1.createdAt).getTime() - new Date(event2.createdAt).getTime();
        });

        return sortedEvents.reduce((videos, event) => {
            event.attach
                .filter(file => file.mimetype.includes('video'))
                .forEach(file => {
                    const [source] = file.filename.split('.');

                    videos.push({
                        id: file.id,
                        eventSaveTime: new Date(file.createdAt).getTime(),
                        source
                    });
                });

            return videos;
        }, []);
    }

    static *getFile(id, type) {
        const options = {
            path: `${config.storagePath}/${id}`,
            encoding: null // Возвращает буфер
        };

        const response = yield this._request(options, type);

        return response.statusCode === 404 ? null : response.body;
    }

    static *getUserData(login) {
        const options = {
            path: `${config.userPath}/${login}`,
            headers: {
                'Content-Type': 'application/json'
            }
        };

        const { body } = yield this._request(options, 'user');

        try {
            return JSON.parse(body);
        } catch (err) {
            log.error('Can not parse body', { err, body });
        }
    }

    static getVideosMarkup(sessionData, videos, isRevision) {
        const metrics = _.get(sessionData, 'timesheet.yaxis', []);
        const metricsIntervals = _.get(sessionData, 'timesheet.xaxis', []);
        const sessionStartDate = _.get(sessionData, 'startedAt');
        const sessionStartTime = new Date(sessionStartDate).getTime();
        const { maxLimitViolation, minVideoDuration } = config;

        const intervalsByVideo = {};

        metrics.forEach((metric, i) => {
            const values = _(metric)
                .keys()
                .filter(violation => metric[violation] >= maxLimitViolation)
                .value();

            if (isRevision || !_.isEmpty(values)) {
                const prevMetricEnd = metricsIntervals[i - 1] ? (metricsIntervals[i - 1] * MS_IN_MINUTE) : 0;
                const metricStart = sessionStartTime + prevMetricEnd;
                const metricEnd = sessionStartTime + (metricsIntervals[i] * MS_IN_MINUTE);

                videos.forEach(video => {
                    const videoStart = video.startTime;
                    const videoEnd = video.startTime + video.duration;

                    const startDiffInMinutes = Math.floor((metricStart - videoStart) / 1000 / 60);
                    const videosDiff = startDiffInMinutes * config.maxVideosDiff;
                    let videoMetricStart = metricStart;

                    // Эта проверка для учета разрывов видео, чтобы не создавать лишних заданий для толокеров
                    if (startDiffInMinutes >= 1) { // 1 min
                        // Так как склеиваем видео, которые длятся меньше минуты, то постепенно набирается
                        // дельта в таймингах видео и метрик
                        videoMetricStart -= videosDiff;
                    }

                    const intervalStart = Math.max(videoMetricStart, videoStart);
                    const intervalEnd = Math.min(metricEnd, videoEnd);

                    if (intervalEnd - intervalStart > minVideoDuration) {
                        const videoName = video.name;

                        intervalsByVideo[videoName] = intervalsByVideo[videoName] || [];
                        intervalsByVideo[videoName].push({
                            start: Math.ceil((intervalStart - videoStart) / 1000),
                            end: Math.floor((intervalEnd - videoStart) / 1000)
                        });
                    }
                });
            }
        });

        return _(intervalsByVideo)
            .keys()
            .map(videoName => ({
                url: S3.getPathToProxyS3('public', videoName, 'videos'),
                intervals: intervalsByVideo[videoName]
            }))
            .value();
    }

    static isMetricsHigh(sessionData) {
        const { technicalMetrics, maxTechMetricsViolation } = config;
        const avgMetrics = _.get(sessionData, 'averages', {});

        return _.some(technicalMetrics, metric => avgMetrics[metric] >= maxTechMetricsViolation);
    }

    static *_concatGroup(group, resultFilePath) {
        const { videoPathsConfig, videoPaths } = group;

        const concatenatedData = {};

        yield new Promise((resolve, reject) => {
            ffmpeg(videoPathsConfig)
                .inputFormat('webm')
                .inputOption(['-f concat', '-safe 0'])
                .outputFormat('webm')
                .outputOptions(['-c copy'])
                .on('end', () => {
                    concatenatedData.path = resultFilePath;

                    removeFiles(videoPaths);
                    fs.unlinkSync(videoPathsConfig);

                    resolve();
                })
                .on('error', error => {
                    reject(error);
                })
                .save(resultFilePath);
        });

        concatenatedData.duration = yield this._getDuration(resultFilePath);

        return concatenatedData;
    }

    // eslint-disable-next-line max-statements
    static *concatVideos(groups, trialId) {
        if (!fs.existsSync(OUTPUT_VIDEO_DIR)) {
            fs.mkdirSync(OUTPUT_VIDEO_DIR);
        }

        const resultVideoByGroupName = {};

        for (const group of groups) {
            const { name } = group;
            const resultFilePath = `${OUTPUT_VIDEO_DIR}/${name}.webm`;

            try {
                resultVideoByGroupName[name] = yield this._concatGroup(group, resultFilePath);
            } catch (error) {
                const inputVideoPaths = _(groups)
                    .map('videoPaths')
                    .flatten()
                    .value();

                const outputVideoPaths = _(resultVideoByGroupName)
                    .values()
                    .map('path')
                    .value();

                removeFiles(_.map(groups, 'videoPathsConfig'));
                removeFiles(inputVideoPaths);
                removeFiles(outputVideoPaths);
                removeFiles([resultFilePath]); // ffmpeg сохраняет пустой файл даже при ошибке склейки

                yield Notifier.failedFfmpegProcessing({ trialId, error: error.message });

                assert(false, 500, 'Concatenate videos was failed', 'CVF', { trialId, error });
            }
        }

        return resultVideoByGroupName;
    }

    static splitVideosIntoGroups(videosData, trialId) {
        const groupsBySource = this._splitVideosIntoGroupsBySource(videosData);

        return _(groupsBySource)
            .keys()
            .map(source => groupsBySource[source])
            .flatten()
            .map(group => {
                const groupStartTime = group[0].startTime;

                return {
                    name: `${trialId}_${uuidv4()}`,
                    videoIds: _.map(group, 'videoId'),
                    videoPaths: _.map(group, 'videoPath'),
                    startTime: groupStartTime,
                    source: group[0].source
                };
            })
            .value();
    }

    static _splitVideosIntoGroupsBySource(videosData) {
        return _(videosData)
            .groupBy('source')
            .mapValues(videosList => {
                const groups = [];
                let currentGroup = [];

                for (const currentVideoData of videosList) {
                    const prevVideoData = _.last(currentGroup);
                    const prevStart = _.get(prevVideoData, 'startTime');
                    const prevDuration = _.get(prevVideoData, 'duration');
                    const prevEnd = prevStart + prevDuration;
                    const videosDistance = Math.abs(currentVideoData.startTime - prevEnd);

                    if (!prevVideoData || videosDistance <= config.maxVideosDiff) {
                        // разница может появляться из-за лагов сети пользователя или браузера/компьютера,
                        // отметка времени ставится в момент прихода запроса на сервер
                        // сейчас 1 "минутное" видео в среднем 58-59 секунд
                        currentGroup.push(currentVideoData);

                        continue;
                    }

                    groups.push(currentGroup);
                    currentGroup = [currentVideoData];
                }

                if (!_.isEmpty(currentGroup)) {
                    groups.push(currentGroup);
                }

                return groups;
            })
            .value();
    }

    static _saveVideosConfig(name, videoIds) {
        const videoPathsConfig = `${INPUT_VIDEO_DIR}/${name}.txt`;

        for (const videoId of videoIds) {
            fs.writeFileSync(videoPathsConfig, `file '${videoId}.webm'\n`, { flag: 'a' });
        }

        return videoPathsConfig;
    }

    static *getConcatenatedVideos(openId, trialId) {
        log.info('Concatenating videos for session', { openId });
        const videosData = yield this.getVideosByOpenId(openId);

        assert(videosData, 424, 'ProctorEdu videos data not loaded', 'VDL', { openId });

        const downloadVideosData = yield this.downloadVideos(videosData, openId);
        const groups = this.splitVideosIntoGroups(downloadVideosData, trialId);

        for (const group of groups) {
            const { name, videoIds } = group;

            group.videoPathsConfig = this._saveVideosConfig(name, videoIds);
        }

        const resultVideosByGroupName = yield this.concatVideos(groups, trialId);

        return groups.map(group => {
            const { name, startTime, source } = group;
            const { path, duration } = resultVideosByGroupName[name];

            return {
                name: `${name}.webm`,
                startTime,
                source,
                duration,
                videoPath: path
            };
        });
    }

    static *downloadVideos(videosData, openId) {
        if (!fs.existsSync(INPUT_VIDEO_DIR)) {
            fs.mkdirSync(INPUT_VIDEO_DIR);
        }

        const downloadVideosData = [];

        for (const videoData of videosData) {
            const { id: videoId, eventSaveTime, source } = videoData;

            const video = yield this.getFile(videoId, 'video');

            assert(video, 424, 'ProctorEdu video not loaded', 'VNL', { openId, videoId });

            try {
                const { videoPath, duration } = yield this._downloadVideoThroughFfmpeg(video, videoId);
                const startTime = eventSaveTime - duration;

                downloadVideosData.push({
                    videoId,
                    startTime,
                    source,
                    videoPath,
                    duration
                });
            } catch (error) {
                removeFiles(_.map(downloadVideosData, 'videoPath'));

                yield Notifier.failedFfmpegProcessing({ openId, videoId, error: error.message });

                assert(false, 500, 'Processing video with ffmpeg was failed', 'PFF', { openId, videoId, error });
            }
        }

        return downloadVideosData;
    }

    static _getDuration(videoPath) {
        return new Promise((resolve, reject) => {
            ffmpeg.ffprobe(videoPath, (err, metadata) => {
                if (err) {
                    reject(err);
                } else {
                    const metaDuration = _.get(metadata, 'format.duration'); // in seconds
                    const duration = Math.floor(metaDuration * 1000) || 0; // in ms

                    resolve(duration);
                }
            });
        });
    }

    static *_downloadVideoThroughFfmpeg(video, videoId) {
        const preparedPath = `${INPUT_VIDEO_DIR}/prepared_${videoId}.webm`;
        const videoPath = `${INPUT_VIDEO_DIR}/${videoId}.webm`;

        fs.writeFileSync(preparedPath, video);

        const downloadVideoData = { videoPath };

        try {
            yield new Promise((resolve, reject) => {
                // https://www.webmproject.org/docs/encoder-parameters/
                // https://trac.ffmpeg.org/wiki/Encode/VP8
                ffmpeg(preparedPath)
                    .inputFormat('webm')
                    .videoCodec('libvpx') // libvpx is the VP8 video encoder for WebM
                    .outputFormat('webm')
                    .outputOptions(['-cpu-used 4', '-crf 33', '-deadline realtime'])
                    .on('end', () => {
                        fs.unlinkSync(preparedPath);

                        resolve();
                    })
                    .on('error', error => {
                        reject(error);
                    })
                    .save(videoPath);
            });

            downloadVideoData.duration = yield this._getDuration(videoPath);
        } catch (err) {
            // ffmpeg сохраняет пустой файл при ошибке, если обработка уже началась
            removeFiles([preparedPath, videoPath]);

            throw err;
        }

        return downloadVideoData;
    }
}

module.exports = ProctorEdu;
