'use strict';

const _ = require('lodash');

const { ProctoringResponses } = require('db/postgres');

const BaseModel = require('models/base');

class ProctoringResponseModel extends BaseModel {
    static *create(trialId, proctoringDataArray, transaction) {
        const attributes = ['source', 'verdict'];

        const dataForCreate = proctoringDataArray.map(proctoringData => _.assign(
            { trialId },
            _.pick(proctoringData, attributes),
            {
                evaluation: _.defaultTo(proctoringData.evaluation, null),
                additionalInfo: _.defaultTo(proctoringData.additionalInfo, null),
                time: Date.now(),
                isLast: false
            }
        ));

        if (!_.isEmpty(dataForCreate)) {
            dataForCreate[dataForCreate.length - 1].isLast = true;
        }

        yield ProctoringResponses.update(
            { isLast: false },
            {
                where: { trialId },
                fields: ['isLast'],
                transaction
            }
        );

        yield ProctoringResponses.bulkCreate(dataForCreate, { transaction });

        return yield this.tryFindLast(trialId, transaction);

    }

    static *tryFindLast(trialId, transaction) {
        return yield ProctoringResponses.findOne({
            where: {
                trialId,
                isLast: true
            },
            attributes: [
                'trialId',
                'source',
                'verdict',
                'evaluation',
                'time',
                'isLast'
            ],
            transaction
        });
    }

    static *setSentToTolokaByTrialIds(trialIds, transaction) {
        return yield ProctoringResponses.update(
            { isSentToToloka: true },
            {
                where: {
                    trialId: { $in: trialIds },
                    isLast: true,
                    $or: [
                        { verdict: 'pending', source: 'proctoring' },
                        { isRevisionRequested: true }
                    ]
                },
                transaction
            }
        );
    }

    static *findByTrialIds(trialIds, transaction) {
        return yield ProctoringResponses.findAll({
            where: { trialId: { $in: trialIds } },
            attributes: [
                'trialId',
                'source',
                'verdict',
                'isSentToToloka',
                'isRevisionRequested',
                'time',
                'isLast'
            ],
            order: 'time',
            raw: true,
            transaction
        });
    }

    static *setIsRevisionRequested(trialId, transaction) {
        yield ProctoringResponses.update(
            { isRevisionRequested: true },
            {
                where: {
                    trialId,
                    isLast: true
                },
                fields: ['isRevisionRequested'],
                transaction
            }
        );
    }

    static *getAdditionalInfo(trialId, transaction) {
        return yield ProctoringResponses.findAll({
            where: {
                trialId,
                additionalInfo: { $ne: null }
            },
            attributes: ['additionalInfo', 'source'],
            raw: true,
            order: [['time']],
            transaction
        });
    }

    static findLastByResponses(proctoringResponses) {
        return proctoringResponses.find(response => response.isLast);
    }
}

module.exports = ProctoringResponseModel;
