'use strict';

const _ = require('lodash');
const assert = require('helpers/assert');
const config = require('yandex-config');

const TrialsComdepReport = require('models/report/items/trialsComdepReport');

class Report {
    constructor(type, query) {
        this._type = type;
        this._query = query;
    }

    static get _items() {
        return [
            /* eslint-disable global-require */
            require('models/report/items/certificateReport'),
            require('models/report/items/certificatesReport'),
            require('models/report/items/certificatesAggregationReport'),
            require('models/report/items/userTrialsReport'),
            require('models/report/items/trialReport'),
            require('models/report/items/trialsReport'),
            require('models/report/items/questionsReport'),
            require('models/report/items/bansByLoginReport'),
            require('models/report/items/bansAggregationReport')
            /* eslint-enable global-require */
        ];
    }

    static get _statItems() {
        return [
            /* eslint-disable global-require */
            require('models/report/items/certificationSummaryReport'),
            require('models/report/items/loginsCertificatesSummaryReport'),
            require('models/report/items/certificatesDetailedReport')
            /* eslint-enable global-require */
        ];
    }

    static getReports(userRoles) {
        return Report._items
            .filter(report => {
                return _.intersection(report.availableRoles, userRoles).length > 0;
            })
            .map(report => ({
                type: report.type,
                description: report.description,
                fields: report.fields
            }));
    }

    *getReportData(userRoles) {
        for (const item of Report._items) {
            if (item.canApply(this._type)) {
                const [userRole] = _.intersection(item.availableRoles, userRoles);

                assert(userRole, 403, 'User has no access to report', 'HNA');

                return yield {
                    data: item.apply(this._query, userRole),
                    blankName: item.getBlankName(userRole)
                };
            }
        }

        assert(false, 400, 'No reporter for type', 'NRF', { type: this._type });
    }

    static *getStatReportsData(from, to) {
        const result = {};

        for (const item of Report._statItems) {
            result[item.type] = yield item.apply({ from, to });
        }

        return result;
    }

    static *getComdepReportData() {
        const reportData = yield TrialsComdepReport.apply();
        const { chunkSize } = config.comdepReport;

        return _.chunk(reportData, chunkSize);
    }
}

module.exports = Report;
