const moment = require('moment');

const BaseReport = require('models/report/items/baseReport');

const assert = require('helpers/assert');
const dateHelper = require('helpers/dateHelper');
const queryHelper = require('helpers/queryHelper');

const { TrialTemplate } = require('db/postgres');

class AggregationBaseReport extends BaseReport {
    static *apply(query) {
        const interval = query.interval || '1m';

        assert(dateHelper.isValidInterval(interval), 400, 'Interval is invalid', 'III', { interval });

        const dates = queryHelper.getInterval(query);
        const totalTo = dates.to;
        let to = dates.from;

        const slugs = yield this._getSlugs(query.slug);
        let result = [];

        do {
            const from = to;

            to = Math.min(this._addInterval(from, interval), totalTo);

            result = result.concat(yield this._getData(from, to, slugs));
        } while (to < totalTo);

        return result;
    }

    static *_getSlugs(slugs) {
        if (slugs) {
            return Array.isArray(slugs) ? slugs : [slugs];
        }

        const exams = yield TrialTemplate.findAll({
            attributes: ['slug'],
            order: 'id',
            raw: true
        });

        return exams.map(exam => exam.slug);
    }

    static _addInterval(from, interval) {
        const measure = dateHelper.getMeasure(interval);
        const count = parseInt(interval, 10);

        const nextDate = moment(from).add(count, measure).toDate();
        const isEndOfMonth = moment(from).date() === moment(from).endOf('month').date();

        if (measure === 'months' && isEndOfMonth) {
            const lastDay = moment(nextDate).endOf('month').date();

            return moment(nextDate).date(lastDay).toDate();
        }

        return nextDate;
    }

    static _getIntervalCondition(from, to) {
        return {
            $and: {
                $gte: from,
                $lte: to
            }
        };
    }
}

module.exports = AggregationBaseReport;
