const _ = require('lodash');

const AggregationBaseReport = require('models/report/items/aggregationBaseReport');

const {
    Ban,
    Certificate,
    Trial,
    TrialTemplate
} = require('db/postgres');

class BansAggregationReport extends AggregationBaseReport {
    static get type() {
        return 'bansAggregation';
    }

    static get description() {
        return 'Сбор статистических данных по банам за указанный период';
    }

    static get fields() {
        return [
            { name: 'from', type: 'date-from', required: true },
            { name: 'to', type: 'date-to', required: false },
            { name: 'interval', type: 'select', required: false },
            { name: 'slug', type: 'array', required: true }
        ];
    }

    static get availableRoles() {
        return ['admin', 'support'];
    }

    static get rolesToBlankName() {
        return {
            default: 'bansAggregation'
        };
    }

    static *_getData(from, to, slugs) {
        const examsInfo = yield this._getExamsInfo(from, to, slugs);
        const bansInfo = yield this._getBansInfo(from, to, slugs);
        const nullifiedCertsInfo = yield this._getNullifiedCertsInfo(from, to, slugs);

        const fromDate = new Date(from);
        const toDate = new Date(to);

        return slugs.reduce((res, slug) => {
            const examInfo = _.find(examsInfo, { slug });
            const examBans = _.find(bansInfo, { slug });

            if (examInfo) {
                const trials = _.get(examInfo, 'trials', []);
                const certificates = _(trials)
                    .map(trial => _.get(trial, 'certificates.0'))
                    .compact()
                    .value();
                const bans = _.get(examBans, 'bans', []);
                const nullifiedCertsByBansCount = _.get(nullifiedCertsInfo, ['examsData', slug, 'ban'], 0);

                res.push({
                    examSlug: slug,
                    from: fromDate,
                    to: toDate,
                    trialsCount: trials.length,
                    certificatesCount: certificates.length,
                    bansCount: bans.length,
                    nullifiedCertsByBansCount,
                    nullifiedOtherCertsCount: nullifiedCertsInfo.certsCount - nullifiedCertsByBansCount
                });
            }

            return res;
        }, []);
    }

    static *_getExamsInfo(from, to, slugs) {
        const includeCertificate = {
            model: Certificate,
            as: 'certificates',
            attributes: ['id']
        };
        const includeTrials = {
            model: Trial,
            where: {
                started: this._getIntervalCondition(from, to),
                nullified: 0
            },
            attributes: ['id'],
            required: false,
            include: includeCertificate
        };

        return yield TrialTemplate.findAll({
            where: { slug: { $in: slugs } },
            attributes: ['slug'],
            include: includeTrials,
            order: [['slug']]
        });
    }

    static *_getNullifiedCertsInfo(from, to, slugs) {
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            where: { slug: { $in: slugs } },
            attributes: ['slug']
        };
        const includeTrial = {
            model: Trial,
            where: { nullified: 0 },
            attributes: ['id'],
            include: includeTrialTemplate
        };
        const certificates = yield Certificate.findAll({
            where: {
                active: 0,
                deactivateDate: this._getIntervalCondition(from, to)
            },
            attributes: ['deactivateReason'],
            include: includeTrial
        });

        const examsData = _(certificates)
            .groupBy('trial.trialTemplate.slug')
            .mapValues(certs => {
                return _(certs)
                    .groupBy('deactivateReason')
                    .mapValues(data => data.length)
                    .value();
            })
            .value();

        return {
            examsData,
            certsCount: certificates.length
        };
    }

    static *_getBansInfo(from, to, slugs) {
        const includeBans = {
            model: Ban,
            as: 'bans',
            where: {
                startedDate: this._getIntervalCondition(from, to),
                action: 'ban'
            },
            attributes: ['id'],
            required: false
        };

        return yield TrialTemplate.findAll({
            where: { slug: { $in: slugs } },
            attributes: ['slug'],
            include: includeBans,
            order: [['slug']]
        });
    }
}

module.exports = BansAggregationReport;
