'use strict';

const _ = require('lodash');

const BaseReport = require('models/report/items/baseReport');

const { getLoginForSearch } = require('helpers/queryHelper');

const {
    Admin,
    Ban,
    GlobalUser,
    TrialTemplate,
    User
} = require('db/postgres');

class BansByLoginReport extends BaseReport {
    static get type() {
        return 'bansByLogin';
    }

    static get description() {
        return 'Получение банов пользователя по логину';
    }

    static get fields() {
        return [{ name: 'login', type: 'string', required: true }];
    }

    static get availableRoles() {
        return ['admin', 'assessor', 'support'];
    }

    static get rolesToBlankName() {
        return {
            default: 'bansByLogin'
        };
    }

    static *apply(query) {
        const { login: searchedLogin } = query;
        const queryLogin = getLoginForSearch(query);

        return yield this._getData(queryLogin, searchedLogin);
    }

    static *_getData(queryLogin, searchedLogin) {
        const user = yield User.findOne({
            where: { login: { $iLike: queryLogin } },
            attributes: ['globalUserId'],
            raw: true
        });

        const globalUserId = _.get(user, 'globalUserId');
        const bansInfo = yield this._getBansInfo(globalUserId);

        return this._bansToJson(bansInfo, searchedLogin);
    }

    static *_getBansInfo(globalUserId) {
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['slug']
        };
        const includeAdmin = {
            model: Admin,
            attributes: ['login']
        };
        const includeBans = {
            model: Ban,
            as: 'bans',
            attributes: [
                'action',
                'startedDate',
                'expiredDate',
                'reason',
                'userLogin'
            ],
            include: [includeTrialTemplate, includeAdmin]
        };
        const includeUsers = {
            model: User,
            as: 'users',
            attributes: ['login']
        };

        return yield GlobalUser.findOne({
            where: { id: globalUserId },
            attributes: ['actualLogin', 'isBanned'],
            include: [includeBans, includeUsers],
            order: [
                [includeBans, includeTrialTemplate, 'id'],
                [includeBans, 'startedDate'],
                [includeUsers, 'login']
            ]
        });
    }

    static _bansToJson(globalUser, searchedLogin) {
        const users = _.get(globalUser, 'users', []);
        const associatedLogins = users
            .map(user => user.login)
            .join(',');
        const bans = _.get(globalUser, 'bans', []);
        const isSuperBanned = this.getValue(globalUser, 'isBanned') ? 'да' : 'нет';
        const actualLogin = this.getValue(globalUser, 'actualLogin');

        return bans.map(ban => ({
            searchedLogin,
            associatedLogins,
            actualLogin,
            isSuperBanned,
            examSlug: this.getValue(ban, 'trialTemplate.slug'),
            action: this.getValue(ban, 'action'),
            adminLogin: this.getValue(ban, 'admin.login'),
            reason: this.getValue(ban, 'reason'),
            startedDate: this.getValue(ban, 'startedDate'),
            expiredDate: this.getValue(ban, 'expiredDate'),
            userLogin: this.getValue(ban, 'userLogin')
        }));
    }
}

module.exports = BansByLoginReport;
