'use strict';
const _ = require('lodash');

const PROCTORING_ANSWER = {
    correct: 1,
    failed: 0,
    pending: '?'
};

const VERDICTS_TO_BIT = {
    correct: 1,
    failed: 0
};

class BaseReport {
    static get type() {
        throw new Error('Base report does not have a type');
    }

    static get description() {
        throw new Error('Base report does not have a description');
    }

    static get fields() {
        throw new Error('Base report does not have a fields');
    }

    static get availableRoles() {
        throw new Error('Base report does not have availableRoles');
    }

    static get rolesToBlankName() {
        throw new Error('Base report does not have rolesToBlankName');
    }

    static get sourceHandlers() {
        return {
            proctoring: (verdict, isRevisionRequested, isSentToToloka) => {
                return {
                    proctoringAnswer: PROCTORING_ANSWER[verdict],
                    isPendingSentToToloka: this._getBit(isSentToToloka && verdict === 'pending'),
                    isRevisionRequested: this._getBit(isRevisionRequested)
                };
            },
            toloka: (verdict, isRevisionRequested) => {
                return {
                    autoTolokaVerdict: VERDICTS_TO_BIT[verdict],
                    isRevisionRequested: this._getBit(isRevisionRequested)
                };
            },
            'toloka-revision': verdict => {
                return { revisionVerdict: VERDICTS_TO_BIT[verdict] };
            },
            appeal: verdict => {
                return { appealVerdict: VERDICTS_TO_BIT[verdict] };
            },
            metrics: () => {
                return { isMetricsHigh: 1 };
            },
            'crit-metrics': () => {
                return { isMetricsHigh: 1 };
            }
        };
    }

    static canApply(type) {
        return this.type === type;
    }

    static apply() {
        throw new Error('Base report can not be applied');
    }

    static getValue(data, path, defaultValue) {
        defaultValue = defaultValue || '';
        const value = _.get(data, path);

        return _.defaultTo(value, defaultValue);
    }

    static getBlankName(userRole) {
        const { roles } = this.rolesToBlankName;

        if (roles && roles[userRole]) {
            return roles[userRole];
        }

        return this.rolesToBlankName.default;
    }

    static _getProctoringData(proctoringResponses, isProctoring, passed) {
        return proctoringResponses.reduce((result, response) => {
            const {
                source,
                verdict,
                isRevisionRequested,
                isSentToToloka
            } = response;

            const processedFields = this.sourceHandlers[source](verdict, isRevisionRequested, isSentToToloka);

            _.assign(result, processedFields);

            if (passed) {
                result.finalVerdict = this._getBit(verdict === 'correct');
            }

            return result;
        }, {
            isProctoring: isProctoring ? 1 : 0,
            proctoringAnswer: '?',
            isMetricsHigh: 0,
            isPendingSentToToloka: 0,
            autoTolokaVerdict: '?',
            isRevisionRequested: 0,
            revisionVerdict: '?',
            appealVerdict: '?',
            finalVerdict: passed
        });
    }

    static _getBit(value) {
        return value ? 1 : 0;
    }
}

module.exports = BaseReport;
