'use strict';

const _ = require('lodash');

const BaseReport = require('models/report/items/baseReport');
const ProctoringResponseModel = require('models/proctoringResponse');

const assert = require('helpers/assert');

const {
    Agency,
    AuthType,
    Certificate,
    Role,
    Trial,
    TrialTemplate,
    User
} = require('db/postgres');

class CertificateReport extends BaseReport {
    static get type() {
        return 'certificate';
    }

    static get description() {
        return 'Получение данных по номеру сертификата';
    }

    static get fields() {
        return [{ name: 'certId', type: 'number', required: true }];
    }

    static get availableRoles() {
        return ['analyst', 'assessor', 'support'];
    }

    static get rolesToBlankName() {
        return {
            default: 'certificate'
        };
    }

    static *apply(query) {
        const { certId } = query;

        assert.isNumber(certId, 400, 'CertId is invalid', 'CII', { certId });

        const cert = yield this._getData(certId);

        assert(cert, 404, 'Certificate not found', 'CNF', { certId });

        const active = this.getValue(cert, 'active');
        const certData = {
            firstname: this.getValue(cert, 'firstname'),
            lastname: this.getValue(cert, 'lastname'),
            login: this.getValue(cert, 'trial.user.login'),
            role: this.getValue(cert, 'trial.user.role.title'),
            authType: this.getValue(cert, 'trial.user.authType.code'),
            examId: this.getValue(cert, 'trial.trialTemplate.id'),
            examTitle: this.getValue(cert, 'trial.trialTemplate.title'),
            certId: this.getValue(cert, 'id'),
            confirmedDate: this.getValue(cert, 'confirmedDate'),
            dueDate: this.getValue(cert, 'dueDate'),
            agencyLogin: this.getValue(cert, 'trial.user.agency.login'),
            agencyManager: _.get(cert, 'trial.user.agency.manager', ''),
            isDeactivated: this._getBit(active === 0),
            deactivateDate: this.getValue(cert, 'deactivateDate'),
            deactivateReason: this.getValue(cert, 'deactivateReason')
        };

        const trialId = _.get(cert, 'trial.id');
        const passed = _.get(cert, 'trial.passed');
        const isProctoring = _.get(cert, 'trial.trialTemplate.isProctoring');

        const proctoringResponses = yield ProctoringResponseModel.findByTrialIds([trialId]);
        const proctoringData = this._getProctoringData(proctoringResponses, isProctoring, passed);

        return _.assign(certData, proctoringData);
    }

    static *_getData(id) {
        const includeAgency = {
            model: Agency,
            attributes: ['login', 'manager'],
            required: false
        };
        const includeRole = {
            model: Role,
            attributes: ['title']
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            attributes: ['id', 'title', 'isProctoring'],
            as: 'trialTemplate'
        };
        const includeAuthType = {
            model: AuthType,
            attributes: ['code'],
            as: 'authType'
        };
        const includeUser = {
            model: User,
            attributes: ['login'],
            include: [includeRole, includeAgency, includeAuthType]
        };
        const includeTrial = {
            model: Trial,
            attributes: ['id', 'trialTemplateId', 'userId', 'passed'],
            include: [includeUser, includeTrialTemplate]
        };

        return yield Certificate.findOne({
            where: { id },
            attributes: [
                'id',
                'dueDate',
                'confirmedDate',
                'firstname',
                'lastname',
                'active',
                'deactivateDate',
                'deactivateReason'
            ],
            include: includeTrial
        });
    }
}

module.exports = CertificateReport;
