'use strict';

const _ = require('lodash');

const ProctoringResponseModel = require('models/proctoringResponse');
const AggregationBaseReport = require('models/report/items/aggregationBaseReport');

const { Certificate, Trial, TrialTemplate } = require('db/postgres');

class CertificatesReport extends AggregationBaseReport {
    static get type() {
        return 'certificatesAggregation';
    }

    static get description() {
        return 'Сбор статистических данных по сертификатам за указанный период';
    }

    static get fields() {
        return [
            { name: 'from', type: 'date-from', required: true },
            { name: 'to', type: 'date-to', required: false },
            { name: 'interval', type: 'select', required: false },
            { name: 'slug', type: 'array', required: true }
        ];
    }

    static get availableRoles() {
        return ['analyst', 'support'];
    }

    static get rolesToBlankName() {
        return {
            default: 'certificatesAggregation'
        };
    }

    static *_getData(from, to, slugs) {
        const exams = yield this._getExams(from, to, slugs);
        const trialIds = _(exams)
            .map('trials')
            .flatten()
            .map('id')
            .value();

        const proctoringResponses = yield ProctoringResponseModel.findByTrialIds(trialIds);
        const proctoringAnswers = _(proctoringResponses)
            .filter(response => response.source === 'proctoring')
            .groupBy('trialId')
            .mapValues(answers => _.last(answers))
            .value();

        return exams.map(exam => {
            const trials = _.get(exam, 'trials', []);

            const data = _(trials)
                .transform((result, trial) => {
                    const verdict = _.get(proctoringAnswers, [trial.id, 'verdict']);
                    const isSentToToloka = _.get(proctoringAnswers, [trial.id, 'isSentToToloka']);

                    if (isSentToToloka) {
                        result.sentToToloka += 1;
                    }

                    if (verdict === 'correct') {
                        result.trialsWithCorrectAnswer.push(trial);
                    } else if (verdict === 'failed') {
                        result.trialsWithFailedAnswer.push(trial);
                    }
                }, {
                    trialsWithCorrectAnswer: [],
                    trialsWithFailedAnswer: [],
                    sentToToloka: 0
                })
                .value();

            const certificatesCount = _.filter(trials, 'certificates.0.active').length;

            return {
                from: new Date(from),
                to: new Date(to),
                slug: exam.slug,
                total: trials.length,
                isProctoring: _.get(exam, 'isProctoring') ? 1 : 0,
                passed: _.sumBy(trials, 'passed'),
                correctAnswersCount: data.trialsWithCorrectAnswer.length,
                failedAnswersCount: data.trialsWithFailedAnswer.length,
                sentToToloka: data.sentToToloka,
                passedWithCorrect: _.sumBy(data.trialsWithCorrectAnswer, 'passed'),
                passedWithFailed: _.sumBy(data.trialsWithFailedAnswer, 'passed'),
                certificatesCount
            };
        });
    }

    static *_getExams(from, to, slugs) {
        const includeCertificate = {
            model: Certificate,
            as: 'certificates',
            attributes: ['active'],
            required: false
        };
        const includeTrials = {
            model: Trial,
            where: {
                started: this._getIntervalCondition(from, to),
                nullified: 0
            },
            attributes: ['id', 'passed'],
            required: false,
            include: includeCertificate
        };

        return yield TrialTemplate.findAll({
            where: { slug: { $in: slugs } },
            attributes: ['slug', 'isProctoring'],
            include: includeTrials,
            order: [['slug']]
        });
    }
}

module.exports = CertificatesReport;
