'use strict';

const moment = require('moment');
const _ = require('lodash');
const Sequelize = require('sequelize');

const { certServices } = require('yandex-config');

const StatBaseReport = require('models/report/items/statBaseReport');

const { Certificate, Trial, TrialTemplate, Service } = require('db/postgres');

const queryHelper = require('helpers/queryHelper');

class CertificatesDetailedReport extends StatBaseReport {
    static get type() {
        return 'certificatesDetailed';
    }

    static get description() {
        return 'Сбор статистических данных по каждому виду сертификации за указанный период';
    }

    static get availableRoles() {
        return ['analyst'];
    }

    static get fields() {
        return [
            { name: 'from', type: 'date-from', required: true },
            { name: 'to', type: 'date-to', required: false }
        ];
    }

    static get rolesToBlankName() {
        return {
            default: 'certificatesDetailed'
        };
    }

    static *apply(query) {
        const { from, to } = queryHelper.getInterval(query);

        const data = yield this._getData(from, to);

        return this._prepareSummaryReport(data, { from, to });
    }

    static _getReportFields() {
        return ['certificatesCount', 'successRate', 'trialsCount'];
    }

    static _toServiceReport(report) {
        const serviceReport = _(report)
            .omit(['serviceCode', 'period'])
            .mapKeys((_val, key) => _.camelCase(`${report.serviceCode}_${key}`))
            .value();
        const successRateKey = _.camelCase(`${report.serviceCode}_successRate`);
        const { certificatesCount, trialsCount } = report;

        serviceReport[successRateKey] =
            certificatesCount ? _.round(certificatesCount / trialsCount * 100) : 0;
        // Убираем лишние пробелы из базы
        serviceReport.period = report.period.split(/\s+/).join(' ');

        return serviceReport;
    }

    static _mergeServiceReports(reports) {
        return _.reduce(reports, _.assign, {});
    }

    static _prepareSummaryReport(reportData, { from, to }) {
        reportData = reportData.map(this._toServiceReport);

        const periods = this._getAllPeriods(from, to);
        const keys = this._getReportFields();
        const servicesKeys = [];

        for (const key of keys) {
            for (const service of certServices) {
                servicesKeys.push(_.camelCase(`${service}_${key}`));
            }
        }

        const hash = _.groupBy(reportData, 'period');

        _.each(hash, (reports, period) => {
            hash[period] = this._mergeServiceReports(reports);
        });

        _(periods).each(period => {
            if (!_.has(hash, period)) {
                hash[period] = { period };
            }

            _(servicesKeys).each(key => {
                if (!_.has(hash[period], key)) {
                    hash[period][key] = 0;
                }
            });
        });

        return _(hash)
            .values()
            .map(report => {
                report.periodDate = this._getFirstDayOfMonth(report.period);
                report.period = this._getLocalizedPeriod(report.period);

                return report;
            })
            .orderBy(({ period }) => moment(period, 'MMMM YYYY'), 'asc')
            .value();
    }

    static *_getData(from, to) {
        const includeService = {
            model: Service,
            where: { code: { $in: certServices } },
            attributes: []
        };
        const includeTrialTemplates = {
            model: TrialTemplate,
            as: 'trialTemplate',
            include: includeService,
            attributes: []
        };
        const includeCertificate = {
            model: Certificate,
            as: 'certificates',
            attributes: [],
            required: false
        };

        return yield Trial.findAll({
            attributes: [
                [
                    Sequelize.fn(
                        'concat',
                        Sequelize.fn('to_char', Sequelize.col('trial.started'), 'Month'),
                        ' ',
                        Sequelize.fn('to_char', Sequelize.col('trial.started'), 'YYYY')
                    ),
                    'period'
                ],
                [Sequelize.fn('count', Sequelize.col('trial.id')), 'trialsCount'],
                [Sequelize.fn('count', Sequelize.col('certificates.id')), 'certificatesCount'],
                [Sequelize.col('trialTemplate.service.code'), 'serviceCode']
            ],
            include: [includeTrialTemplates, includeCertificate],
            where: {
                started: {
                    $and: {
                        $gte: from,
                        $lte: to
                    }
                },
                nullified: 0
            },
            group: ['period', 'serviceCode'],
            order: [[Sequelize.col('period'), 'DESC']],
            raw: true
        });
    }
}

module.exports = CertificatesDetailedReport;
