'use strict';

const _ = require('lodash');
const config = require('yandex-config');

const BaseReport = require('models/report/items/baseReport');
const ProctoringResponseModel = require('models/proctoringResponse');
const BlackboxModel = require('models/blackbox');

const queryHelper = require('helpers/queryHelper');

const {
    Certificate,
    Trial,
    TrialTemplate,
    User,
    AuthType
} = require('db/postgres');

class CertificatesReport extends BaseReport {
    static get type() {
        return 'certificates';
    }

    static get description() {
        return 'Получение всех сертификатов за указанный период';
    }

    static get fields() {
        return [
            { name: 'from', type: 'date-from', required: true },
            { name: 'to', type: 'date-to', required: false },
            { name: 'slug', type: 'array', required: true }
        ];
    }

    static get availableRoles() {
        return ['analyst', 'support'];
    }

    static get rolesToBlankName() {
        return {
            default: 'certificates'
        };
    }

    static *apply(query) {
        const interval = queryHelper.getInterval(query);
        const slug = queryHelper.getArray(query, 'slug');

        const certs = yield this._getData(interval.from, interval.to, slug);

        const trialIds = _.map(certs, 'trial.id');
        const uids = _(certs)
            .map('trial.user.uid')
            .uniq()
            .value();

        const blackbox = new BlackboxModel(config.blackbox);
        const emails = yield blackbox.getEmails(uids);

        const proctoringResponses = yield ProctoringResponseModel.findByTrialIds(trialIds);
        const proctoringResponsesByTrialId = _.groupBy(proctoringResponses, 'trialId');

        return certs.map(this._certificateToJson.bind(this, proctoringResponsesByTrialId, emails));
    }

    static *_getData(from, to, slug) {
        const includeAuthType = {
            model: AuthType,
            attributes: ['code'],
            as: 'authType'
        };
        const includeUser = {
            model: User,
            attributes: ['uid', 'login'],
            include: includeAuthType
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            where: slug.length ? { slug: { in: slug } } : {},
            attributes: ['title', 'slug', 'isProctoring'],
            as: 'trialTemplate'
        };
        const includeTrial = {
            model: Trial,
            where: { nullified: 0 },
            attributes: ['id', 'nullified', 'passed'],
            include: [includeTrialTemplate, includeUser]
        };

        return yield Certificate.findAll({
            where: {
                $and: [
                    { confirmedDate: { gte: from } },
                    { confirmedDate: { lte: to } }
                ]
            },
            attributes: ['id', 'confirmedDate', 'firstname', 'lastname'],
            include: includeTrial,
            order: [['confirmedDate']]
        });
    }

    static _certificateToJson(proctoringResponses, emails, data) {
        const uid = this.getValue(data, 'trial.user.uid', '');
        const certData = {
            certId: this.getValue(data, 'id', ''),
            confirmedDate: this.getValue(data, 'confirmedDate', ''),
            authType: this.getValue(data, 'trial.user.authType.code', ''),
            examTitle: this.getValue(data, 'trial.trialTemplate.title', ''),
            examSlug: this.getValue(data, 'trial.trialTemplate.slug', ''),
            firstname: this.getValue(data, 'firstname', ''),
            lastname: this.getValue(data, 'lastname', ''),
            login: this.getValue(data, 'trial.user.login', ''),
            email: this.getValue(emails, uid, '')
        };

        const trialId = _.get(data, 'trial.id');
        const passed = _.get(data, 'trial.passed');
        const isProctoring = _.get(data, 'trial.trialTemplate.isProctoring');
        const responsesByTrial = proctoringResponses[trialId] || [];

        const proctoringData = this._getProctoringData(responsesByTrial, isProctoring, passed);

        return _.assign(certData, proctoringData);
    }
}

module.exports = CertificatesReport;
