'use strict';

const _ = require('lodash');
const Sequelize = require('sequelize');

const StatBaseReport = require('models/report/items/statBaseReport');

const { Certificate, Trial, TrialTemplate, Type } = require('db/postgres');

const queryHelper = require('helpers/queryHelper');

class CertificationSummaryReport extends StatBaseReport {
    static get type() {
        return 'certificationSummary';
    }

    static get description() {
        return 'Сбор статистических данных по всем видам сертификации за указанный период';
    }

    static get availableRoles() {
        return ['analyst'];
    }

    static get fields() {
        return [
            { name: 'from', type: 'date-from', required: true },
            { name: 'to', type: 'date-to', required: false }
        ];
    }

    static get rolesToBlankName() {
        return {
            default: 'certificationSummary'
        };
    }

    static get emptyReport() {
        return {
            certificatesCount: 0,
            successRate: 0,
            trialsCount: 0
        };
    }

    static *apply(query) {
        const { from, to } = queryHelper.getInterval(query);

        const certificationData = yield this._getData(from, to);

        certificationData.forEach(trialData => {
            const { trialsCount, certificatesCount, period } = trialData;

            trialData.successRate = certificatesCount ? _.round(certificatesCount / trialsCount * 100) : 0;
            trialData.period = this._getLocalizedPeriod(period);
        });

        return certificationData;
    }

    static *_getTrialsData(from, to) {
        const includeTypes = {
            model: Type,
            attributes: [],
            required: true,
            where: {
                code: 'cert'
            }
        };

        const includeTrialTemplates = {
            model: TrialTemplate,
            as: 'trialTemplate',
            include: includeTypes,
            attributes: []
        };

        return yield Trial.findAll({
            attributes: [
                [
                    Sequelize.fn(
                        'concat',
                        Sequelize.fn('to_char', Sequelize.col('trial.started'), 'Month'),
                        ' ',
                        Sequelize.fn('to_char', Sequelize.col('trial.started'), 'YYYY')
                    ),
                    'period'
                ],
                [Sequelize.fn('count', Sequelize.col('trial.id')), 'trialsCount']
            ],
            include: includeTrialTemplates,
            where: {
                started: {
                    $and: {
                        $gte: from,
                        $lte: to
                    }
                },
                nullified: 0
            },
            group: ['period'],
            order: [[Sequelize.col('period'), 'DESC']],
            raw: true
        });
    }

    static *_getCertificatesData(from, to) {
        const includeTypes = {
            model: Type,
            attributes: [],
            where: {
                code: 'cert'
            }
        };

        const includeCertificate = {
            model: Certificate,
            as: 'certificates',
            attributes: [],
            include: includeTypes,
            required: true
        };

        return yield Trial.findAll({
            attributes: [
                [
                    Sequelize.fn(
                        'concat',
                        Sequelize.fn('to_char', Sequelize.col('trial.started'), 'Month'),
                        ' ',
                        Sequelize.fn('to_char', Sequelize.col('trial.started'), 'YYYY')
                    ),
                    'period'
                ],
                [Sequelize.fn('count', Sequelize.col('certificates.id')), 'certificatesCount']
            ],
            include: includeCertificate,
            where: {
                started: {
                    $and: {
                        $gte: from,
                        $lte: to
                    }
                },
                nullified: 0
            },
            group: ['period'],
            order: [[Sequelize.col('period'), 'DESC']],
            raw: true
        });
    }

    static *_getData(from, to) {
        const [
            certificatesData,
            trialsData
        ] = yield [
            this._getCertificatesData(from, to),
            this._getTrialsData(from, to)
        ];

        const reportData = _.merge(certificatesData, trialsData);

        // Убираем лишние пробелы из базы
        reportData.forEach(report => {
            report.period = report.period.split(/\s+/).join(' ');
            report.periodDate = this._getFirstDayOfMonth(report.period);
        });

        return this._prepareSummaryReport(reportData, { from, to });
    }
}

module.exports = CertificationSummaryReport;
