const _ = require('lodash');
const Sequelize = require('sequelize');

const StatBaseReport = require('models/report/items/statBaseReport');

const {
    Certificate,
    Service,
    Trial,
    TrialTemplate,
    Type,
    User
} = require('db/postgres');

const queryHelper = require('helpers/queryHelper');

class LoginsCertificatesSummaryReport extends StatBaseReport {
    static get type() {
        return 'loginsCertificatesSummary';
    }

    static get description() {
        return 'Сбор статистических данных по логинам с сертификатами за указанный период';
    }

    static get availableRoles() {
        return ['analyst'];
    }

    static get fields() {
        return [
            { name: 'from', type: 'date-from', required: true },
            { name: 'to', type: 'date-to', required: false }
        ];
    }

    static get rolesToBlankName() {
        return {
            default: 'logins'
        };
    }

    static get emptyReport() {
        return {
            loginsWithCerts: 0,
            loginsWithSeveralCerts: 0,
            multiCertsRate: 0,
            popularCertsCombination: ''
        };
    }

    static *apply(query) {
        const { from, to } = queryHelper.getInterval(query);

        const certsData = yield this._getData(from, to);
        const dataByPeriods = this._getDataByPeriods(certsData);

        const data = this._prepareSummaryReport(dataByPeriods, { from, to });

        data.forEach(item => {
            item.period = this._getLocalizedPeriod(item.period);
        });

        return data;
    }

    static _getDataByPeriods(certsData) {
        const groups = _.groupBy(certsData, 'period');

        return Object.keys(groups).map(period => {
            const certsByUid = _.groupBy(groups[period], 'uid');
            const loginsWithCerts = Object.keys(certsByUid).length;
            const loginsWithSeveralCerts = _(certsByUid)
                .keys()
                .filter(uid => certsByUid[uid].length > 1)
                .size();

            return {
                period,
                periodDate: this._getFirstDayOfMonth(period),
                loginsWithCerts,
                loginsWithSeveralCerts,
                multiCertsRate: _.round(loginsWithSeveralCerts / loginsWithCerts * 100),
                popularCertsCombination: this._getPopularCombination(certsByUid)
            };
        });
    }

    static _getCombinations(services) {
        return services
            .reduce((subsets, service) => {
                return subsets.concat(subsets.map(subset => [service, ...subset]));
            }, [[]])
            .filter(combination => combination.length > 1)
            .map(combination => combination.sort().join(' + '));
    }

    static _getPopularCombination(certsByUid) {
        const allCombinations = _(certsByUid)
            .keys()
            .filter(uid => certsByUid[uid].length > 1)
            .map(uid => certsByUid[uid].map(cert => _.get(cert, 'service')))
            .map(services => this._getCombinations(services))
            .flatten()
            .value();

        if (_.isEmpty(allCombinations)) {
            return '';
        }

        const countCombinations = _.countBy(allCombinations, item => item);

        return _.orderBy(allCombinations, [
            item => countCombinations[item],
            item => item.split(' + ').length
        ], ['desc', 'desc'])[0];
    }

    static *_getData(from, to) {
        const includeService = {
            model: Service,
            attributes: ['title']
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['id'],
            include: [includeService]
        };
        const includeUser = {
            model: User,
            where: { yandexUid: { $ne: null } },
            attributes: ['yandexUid']
        };

        const includeTrials = {
            model: Trial,
            where: { nullified: 0 },
            attributes: ['id'],
            include: [includeUser, includeTrialTemplate]
        };
        const includeType = {
            model: Type,
            where: { code: 'cert' },
            attributes: []
        };
        const certs = yield Certificate.findAll({
            where: {
                confirmedDate: {
                    $and: {
                        $gte: from,
                        $lte: to
                    }
                }
            },
            attributes: [
                [
                    Sequelize.fn(
                        'concat',
                        Sequelize.fn('to_char', Sequelize.col('confirmed_date'), 'Month'),
                        ' ',
                        Sequelize.fn('to_char', Sequelize.col('confirmed_date'), 'YYYY')
                    ),
                    'period'
                ]
            ],
            include: [includeType, includeTrials]
        });

        return certs.map(cert => ({
            period: cert.get('period').replace(/\s+/, ' '),
            service: _.get(cert, 'trial.trialTemplate.service.title', ''),
            uid: _.get(cert, 'trial.user.yandexUid', '')
        }));
    }
}

module.exports = LoginsCertificatesSummaryReport;
