const BaseReport = require('models/report/items/baseReport');

const assert = require('helpers/assert');
const queryHelper = require('helpers/queryHelper');

const db = require('db/postgres');
const { TrialToQuestion, Question, Trial, TrialTemplate } = db;

class QuestionsReport extends BaseReport {
    static get type() {
        return 'questions';
    }

    static get description() {
        return 'Получение статистики по ответам на вопросы';
    }

    static get fields() {
        return [
            { name: 'slug', type: 'string', required: true },
            { name: 'from', type: 'date-from', required: true },
            { name: 'to', type: 'date-to', required: false }
        ];
    }

    static get availableRoles() {
        return ['analyst', 'support'];
    }

    static get rolesToBlankName() {
        return {
            default: 'questions'
        };
    }

    static *apply(query) {
        const interval = queryHelper.getInterval(query);
        const data = yield this._getData(query.slug, interval.from, interval.to);

        return this._dataToJson(data);
    }

    static *_getData(slug, from, to) {
        const trialTemplate = yield this._getTrialTemplate(slug, from, to);

        assert(trialTemplate, 404, 'Test not found', 'TNF', { slug });

        const trialIds = trialTemplate.trials.map(trial => trial.id);
        const questions = yield this._getQuestions(trialIds);

        return {
            examSlug: slug,
            questions
        };
    }

    static *_getQuestions(trialIds) {
        const includeTrialsToQuestions = {
            model: TrialToQuestion,
            as: 'trialToQuestions',
            where: { trialId: { $in: trialIds }, answered: { $gt: 0 } },
            attributes: []
        };

        return yield Question.findAll({
            where: { active: 1 },
            attributes: [
                'id',
                'type',
                'text',
                [db.sequelize.fn('count', db.sequelize.col('trialToQuestions.answered')), 'askedCount'],
                [db.sequelize.fn('sum', db.sequelize.col('trialToQuestions.correct')), 'correctCount']
            ],
            include: includeTrialsToQuestions,
            group: ['id', 'type', 'text'],
            order: ['id']
        });
    }

    static *_getTrialTemplate(slug, from, to) {
        const includeTrials = {
            model: Trial,
            as: 'trials',
            where: { started: { $gte: from, $lte: to } },
            attributes: ['id'],
            required: false
        };

        return yield TrialTemplate.findOne({
            where: { slug },
            attributes: ['id', 'slug'],
            include: includeTrials
        });
    }

    static _dataToJson(data) {
        const { examSlug, questions } = data;

        return questions.map(question => ({
            examSlug,
            questionId: this.getValue(question, 'id'),
            type: question.type ? 'несколько' : 'один',
            text: this.getValue(question, 'text'),
            askedCount: question.get('askedCount'),
            correctCount: question.get('correctCount')
        }));
    }
}

module.exports = QuestionsReport;
