'use strict';

const BaseReport = require('models/report/items/baseReport');
const AttemptModel = require('models/attempt');

const assert = require('helpers/assert');

const {
    Question,
    Section,
    Trial,
    TrialTemplate,
    TrialToQuestion,
    User
} = require('db/postgres');

class TrialReport extends BaseReport {
    static get type() {
        return 'trial';
    }

    static get description() {
        return 'Получение данных о попытке';
    }

    static get fields() {
        return [{ name: 'trialId', type: 'number', required: true }];
    }

    static get availableRoles() {
        return ['analyst', 'support', 'assessor'];
    }

    static get rolesToBlankName() {
        return {
            roles: { assessor: 'trialAssessor' },
            default: 'trial'
        };
    }

    static get additionalFieldsByRoles() {
        return {
            analyst: ['text'],
            support: ['text']
        };
    }

    static get fieldsHandlers() {
        return {
            text: answer => this.getValue(answer, 'question.text')
        };
    }

    static *apply(query, userRole) {
        const { trialId } = query;

        assert.isNumber(trialId, 400, 'Trial id not a number', 'TNN', { trialId });

        const data = yield this._getData(trialId);

        return this._answersToJson(data, userRole);
    }

    static *_getData(trialId) {
        const includeUser = {
            model: User,
            attributes: ['login']
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['title']
        };
        const includeSections = {
            model: Section,
            attributes: ['title']
        };
        const questionsCondition = yield AttemptModel.getQuestionsCondition(trialId);
        const includeQuestion = {
            model: Question,
            where: questionsCondition,
            attributes: ['text'],
            include: includeSections
        };

        return yield [
            Trial.findOne({
                where: { id: trialId },
                attributes: [],
                include: [includeUser, includeTrialTemplate]
            }),
            TrialToQuestion.findAll({
                where: { trialId },
                attributes: ['seq', 'answered', 'correct'],
                include: includeQuestion,
                order: 'seq'
            })
        ];
    }

    static _answersToJson(data, userRole) {
        const login = this.getValue(data, '0.user.login', '');
        const exam = this.getValue(data, '0.trialTemplate.title', '');

        return data[1].map(answer => {
            const questionData = {
                login,
                exam,
                section: this.getValue(answer, 'question.section.title', ''),
                seq: this.getValue(answer, 'seq', ''),
                answered: this.getValue(answer, 'answered', ''),
                correct: this.getValue(answer, 'correct', '')
            };

            const additionalFields = this.additionalFieldsByRoles[userRole];

            if (additionalFields) {
                for (const field of additionalFields) {
                    questionData[field] = this.fieldsHandlers[field](answer);
                }
            }

            return questionData;
        });
    }
}

module.exports = TrialReport;
