'use strict';

const _ = require('lodash');
const config = require('yandex-config');
const moment = require('moment');

const BaseReport = require('models/report/items/baseReport');
const BlackboxModel = require('models/blackbox');

const {
    Certificate,
    ProctoringResponses,
    Trial,
    TrialTemplate,
    User
} = require('db/postgres');

class TrialsComdepReport extends BaseReport {
    static get type() {
        return 'trialsComdep';
    }

    static get description() {
        return 'Получение данных о попытках для КД';
    }

    static get fields() {
        return [];
    }

    static get availableRoles() {
        return ['analyst'];
    }

    static get rolesToBlankName() {
        return {
            default: 'trialsComdep'
        };
    }

    static *apply() {
        const { from, examSlugs } = config.comdepReport;

        const trialsData = yield this._getData(from, examSlugs);
        const uids = _(trialsData)
            .map('user.uid')
            .uniq()
            .value();

        const blackbox = new BlackboxModel(config.blackbox);
        const emailsByUids = yield blackbox.getEmails(uids);

        return trialsData.map(this._trialToJson.bind(this, emailsByUids));
    }

    static *_getData(from, examSlugs) {
        const includeCertificate = {
            model: Certificate,
            as: 'certificates',
            attributes: ['id', 'active']
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            where: { slug: examSlugs },
            attributes: []
        };
        const includeProctoringResponses = {
            model: ProctoringResponses,
            where: { isLast: true },
            attributes: [
                'verdict',
                'isSentToToloka',
                'isRevisionRequested'
            ],
            as: 'proctoringResponses'
        };
        const includeUser = {
            model: User,
            attributes: [
                'uid',
                'login',
                'firstname',
                'lastname'
            ]
        };

        return yield Trial.findAll({
            attributes: ['id', 'started', 'passed'],
            where: {
                started: { $gte: from },
                nullified: 0,
                expired: 1
            },
            include: [
                includeCertificate,
                includeProctoringResponses,
                includeTrialTemplate,
                includeUser
            ],
            order: [['started']]
        });
    }

    // eslint-disable-next-line complexity
    static _getFinalVerdict(trial, certId) {
        const { passed, proctoringResponses } = trial;

        if (!passed) {
            return 'failure';
        }

        const [{ verdict, isSentToToloka, isRevisionRequested }] = proctoringResponses;

        if (isSentToToloka || isRevisionRequested || verdict === 'pending') {
            return 'verification';
        }

        return (verdict === 'correct' && certId) ? 'success' : 'failure';
    }

    static _getCertId(trial) {
        const certId = _.get(trial, 'certificates.0.id', '');
        const certActive = _.get(trial, 'certificates.0.active');

        if (!certId || !certActive) {
            return '';
        }

        return certId.toString();
    }

    static _trialToJson(emailsByUids, trial) {
        const uid = _.get(trial, 'user.uid');
        const email = _.get(emailsByUids, uid, '');
        const certId = this._getCertId(trial);

        return {
            login: _.get(trial, 'user.login'),
            firstname: this.getValue(trial, 'user.firstname'),
            lastname: this.getValue(trial, 'user.lastname'),
            email,
            trialId: trial.id,
            date: moment(trial.started).format('DD.MM.YYYY'),
            finalVerdict: this._getFinalVerdict(trial, certId),
            certId
        };
    }
}

module.exports = TrialsComdepReport;
