'use strict';

const _ = require('lodash');
const moment = require('moment');
const config = require('yandex-config');

const BaseReport = require('models/report/items/baseReport');
const ProctoringResponseModel = require('models/proctoringResponse');
const BlackboxModel = require('models/blackbox');

const queryHelper = require('helpers/queryHelper');
const assert = require('helpers/assert');

const {
    AuthType,
    Certificate,
    Trial,
    TrialTemplate,
    User
} = require('db/postgres');

class TrialsReport extends BaseReport {
    static get type() {
        return 'trials';
    }

    static get description() {
        return 'Получение данных о попытках за указанный период';
    }

    static get fields() {
        return [
            { name: 'from', type: 'date-from', required: true },
            { name: 'to', type: 'date-to', required: false },
            { name: 'slug', type: 'select', required: true },
            { name: 'login', type: 'array', required: false }
        ];
    }

    static get availableRoles() {
        return ['analyst', 'support'];
    }

    static get rolesToBlankName() {
        return {
            default: 'trials'
        };
    }

    static *apply(query) {
        const interval = queryHelper.getInterval(query);
        const logins = queryHelper.getArray(query, 'login');
        const { slug } = query;

        assert.isSlug(slug, 400, 'Exam slug contains invalid characters', 'EIC', { slug });

        const conditions = {
            from: interval.from,
            to: interval.to,
            logins,
            slug
        };

        const users = yield this._getData(conditions);

        const uids = _.map(users, 'uid');

        const blackbox = new BlackboxModel(config.blackbox);
        const emails = yield blackbox.getEmails(uids);

        const trialIds = _(users)
            .map(user => {
                const trials = _.get(user, 'trials', []);

                return _.map(trials, 'id');
            }, [])
            .flatten()
            .value();

        const proctoringResponses = yield ProctoringResponseModel.findByTrialIds(trialIds);
        const responsesByTrialId = _.groupBy(proctoringResponses, 'trialId');

        return this._usersToJson(responsesByTrialId, users, emails);
    }

    static *_getData(conditions) {
        const includeCertificate = {
            model: Certificate,
            as: 'certificates',
            attributes: ['id']
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            where: { slug: conditions.slug },
            attributes: ['isProctoring']
        };
        const includeTrials = {
            model: Trial,
            as: 'trials',
            where: {
                started: {
                    $and: {
                        $gte: conditions.from,
                        $lte: conditions.to
                    }
                },
                nullified: 0
            },
            attributes: ['id', 'started', 'passed'],
            include: [includeCertificate, includeTrialTemplate]
        };
        const includeAuthType = {
            model: AuthType,
            as: 'authType',
            attributes: ['code'],
            required: true
        };

        const query = {
            attributes: ['login', 'uid', 'firstname', 'lastname'],
            include: [includeTrials, includeAuthType],
            order: [
                ['login'],
                [includeTrials, 'started', 'DESC']
            ]
        };

        const usersQuery = this._getUsersQuery(conditions.logins);

        if (usersQuery) {
            query.where = usersQuery;
        }

        return yield User.findAll(query);
    }

    static _getUsersQuery(logins) {
        if (_.isEmpty(logins)) {
            return;
        }

        const loginRegExp = new RegExp('[-.]+', 'g');

        const $or = logins.map(login => {
            assert(/^[.a-z0-9_-]+$/i.test(login), 400, 'Login contains invalid characters', 'LIC', { login });

            const queryLogin = login.replace(loginRegExp, '_');

            return { login: { $like: queryLogin } };
        });

        return { $or };
    }

    static _usersToJson(proctoringResponses, users, emails) {
        const usersData = users.map(user => {
            const userData = {
                login: this.getValue(user, 'login'),
                authType: this.getValue(user, 'authType.code'),
                firstname: this.getValue(user, 'firstname'),
                lastname: this.getValue(user, 'lastname'),
                email: this.getValue(emails, user.uid)
            };

            return user
                .get('trials')
                .map(this._trialToJson.bind(this, proctoringResponses, userData));
        });

        return _(usersData)
            .flatten()
            .value();
    }

    static _trialToJson(proctoringResponses, userData, trial) {
        const date = this.getValue(trial, 'started', '');
        const trialId = _.get(trial, 'id');
        const isProctoring = _.get(trial, 'trialTemplate.isProctoring');
        const passed = _.get(trial, 'passed');
        const { login, authType, email, firstname, lastname } = userData;

        const trialData = {
            login,
            firstname,
            lastname,
            authType,
            email,
            trialId,
            passed,
            date: moment(date).format('DD.MM.YYYY'),
            certId: this.getValue(trial, 'certificates.0.id', '')
        };

        const responsesByTrial = proctoringResponses[trialId] || [];
        const proctoringData = this._getProctoringData(responsesByTrial, isProctoring, passed);

        return _.assign(trialData, proctoringData);
    }
}

module.exports = TrialsReport;
