'use strict';

const _ = require('lodash');

const BaseReport = require('models/report/items/baseReport');
const ProctoringResponseModel = require('models/proctoringResponse');

const { getLoginForSearch } = require('helpers/queryHelper');

const {
    Trial,
    User,
    Certificate,
    TrialTemplate
} = require('db/postgres');

class UserTrialsReport extends BaseReport {
    static get type() {
        return 'userTrials';
    }

    static get description() {
        return 'Получение попыток пользователя по логину';
    }

    static get fields() {
        return [{ name: 'login', type: 'string', required: true }];
    }

    static get availableRoles() {
        return ['analyst', 'assessor', 'support'];
    }

    static get rolesToBlankName() {
        return {
            default: 'userTrials'
        };
    }

    static *apply(query) {
        const queryLogin = getLoginForSearch(query);
        const trials = yield this._getData(queryLogin);

        const trialIds = _.map(trials, 'id');
        const proctoringResponses = yield ProctoringResponseModel.findByTrialIds(trialIds);
        const responsesByTrialId = _.groupBy(proctoringResponses, 'trialId');

        return trials.map(this._trialToJson.bind(this, responsesByTrialId));
    }

    static *_getData(login) {
        const includeTrialTemplate = {
            model: TrialTemplate,
            attributes: ['title', 'isProctoring'],
            as: 'trialTemplate'
        };
        const includeCertificate = {
            model: Certificate,
            attributes: ['id', 'dueDate', 'active', 'deactivateDate', 'deactivateReason'],
            as: 'certificates',
            required: false
        };
        const includeUser = {
            model: User,
            where: { login: { $iLike: login } },
            attributes: ['login']
        };

        return yield Trial.findAll({
            attributes: ['id', 'started', 'finished', 'passed', 'nullified'],
            include: [includeUser, includeCertificate, includeTrialTemplate],
            order: 'finished'
        });
    }

    static _trialToJson(proctoringResponses, trial) {
        const trialId = _.get(trial, 'id');
        const passed = _.get(trial, 'passed');
        const certActive = this.getValue(trial, 'certificates.0.active');

        const trialData = {
            trialId,
            passed,
            exam: this.getValue(trial, 'trialTemplate.title'),
            started: this.getValue(trial, 'started'),
            finished: this.getValue(trial, 'finished'),
            nullified: this.getValue(trial, 'nullified'),
            certId: this.getValue(trial, 'certificates.0.id'),
            certDueDate: this.getValue(trial, 'certificates.0.dueDate'),
            isCertDeactivated: certActive === '' ? '' : this._getBit(certActive === 0),
            deactivateDate: this.getValue(trial, 'certificates.0.deactivateDate'),
            deactivateReason: this.getValue(trial, 'certificates.0.deactivateReason')
        };

        const isProctoring = _.get(trial, 'trialTemplate.isProctoring');
        const responsesByTrial = proctoringResponses[trialId] || [];
        const proctoringData = this._getProctoringData(responsesByTrial, isProctoring, passed);

        return _.assign(trialData, proctoringData);
    }
}

module.exports = UserTrialsReport;
