'use strict';

const log = require('logger');

const assert = require('helpers/assert');
const AWS = require('aws-sdk');
const config = require('yandex-config').s3;
const Notifier = require('models/notifier');

const BUCKET_TYPES = {
    videos: 'public',
    faces: 'public',
    documents: 'private'
};

class S3 {
    constructor(credentials) {
        this._bucket = credentials.bucket;
        this._client = new AWS.S3(credentials);
    }

    *_tryPutObject(options, tryCount = 1) {
        try {
            return yield this._putObject(options);
        } catch (error) {
            if (tryCount <= config.maxRepeatCount) {
                return yield this._tryPutObject(options, tryCount + 1);
            }

            yield Notifier.failedS3Notify(error);

            log.error('Failed put object to S3', { error });

            assert(false, 424, 'Data not loaded to S3', 'DNL', {
                error,
                bucket: options.Bucket,
                key: options.Key
            });
        }
    }

    _putObject(options) {
        return new Promise((resolve, reject) => {
            this._client.putObject(options, (error, data) => {
                if (error || !data) {
                    error = error || new Error('No data');
                    reject(error);
                }

                resolve(data);
            });
        });
    }

    *upload(file, key, access) {
        key = key || Date.now().toString();

        assert(file instanceof Buffer, 500, 'File is not a buffer', 'FNB', { bucket: this._bucket, key });

        const options = {
            Bucket: this._bucket,
            Key: `${config.env}/${key}`,
            Body: file,
            ACL: access || 'public-read'
        };

        return yield this._tryPutObject(options);
    }

    static getPathToProxyS3(bucketType, fileName, path) {
        if (!fileName) {
            return '';
        }

        const bucket = config.buckets[bucketType];
        const prefix = `${config.proxyPath}/${bucket}/${config.env}`;

        return path ? `${prefix}/${path}/${fileName}` : `${prefix}/${fileName}`;
    }

    static getPathToS3Read(dir, filename) {
        const bucketType = BUCKET_TYPES[dir];
        const bucket = config.buckets[bucketType];

        return `${config.endpoint}/${bucket}/${config.env}/${dir}/${filename}`;
    }
}

module.exports = S3;
