'use strict';

const { User, AuthType, Trial } = require('db/postgres');

const Base = require('models/base');

const _ = require('lodash');

class BaseUserModel extends Base {
    static *findAndRenew(userState, transaction) {
        const uid = _.chain(userState)
            .get('uid.value')
            .parseInt()
            .value();

        const data = yield {
            storedUser: this.findUser({ where: { uid }, transaction }),
            authType: AuthType.findOne({
                where: { code: this.authType },
                attributes: ['id'],
                transaction
            })
        };
        const authTypeId = data.authType.get('id');
        const roleId = _.get(data.storedUser, 'roleId', 1);
        const userData = _.assign(
            { uid, roleId, authTypeId },
            this._getUserData(userState, data.storedUser)
        );

        this._setUserData(userData, uid);

        yield User.upsert(userData, { transaction });
        const result = yield this.findUser({ where: { uid }, transaction });

        return new this(result);
    }

    static _setUserData() {
        throw new Error('Base model can not set user data');
    }

    static *findUser(data) {
        const { where, attributes, transaction } = data;

        const includeAuthType = {
            model: AuthType,
            where: { code: this.authType },
            attributes: ['code'],
            as: 'authType'
        };

        const query = {
            where,
            include: includeAuthType,
            attributes,
            transaction
        };

        return yield User.findOne(query);
    }

    static *findUserByTrialId(data, transaction) {
        const includeTrial = {
            model: Trial,
            as: 'trials',
            where: { id: data.trialId },
            attributes: []
        };

        return yield User.findOne({
            include: [includeTrial],
            attributes: data.attributes,
            raw: true,
            transaction
        });
    }

    static *setGlobalUserId({ users, globalUserId, transaction }) {
        users = users || [];
        const ids = _.map(users, 'id');

        yield User.update({
            globalUserId
        }, {
            where: { id: { $in: ids } },
            fields: ['globalUserId'],
            transaction
        });
    }

    static *getAssociatedUsers(globalUserIds, transaction) {
        const includeAuthType = {
            model: AuthType,
            where: { code: this.authType },
            attributes: [],
            as: 'authType'
        };

        return yield User.findAll({
            where: { globalUserId: globalUserIds },
            include: includeAuthType,
            attributes: ['id', 'login'],
            order: [['id']],
            raw: true,
            transaction
        });
    }
}

module.exports = BaseUserModel;
