'use strict';

const { self } = require('yandex-config');

const _ = require('lodash');
const url = require('url');
const path = require('path');

const assert = require('helpers/assert');

const BaseUserModel = require('models/user/baseUserModel');

const {
    AuthType,
    Certificate,
    ProctoringVideos,
    Trial,
    TrialTemplate,
    User,
    UserIdentification
} = require('db/postgres');

const SPACE_REG = new RegExp('&nbsp;', 'g');
const IS_EMPTY_VALUE = value => _.isNil(value) || (_.isArray(value) && _.isEmpty(value));

class WebUser extends BaseUserModel {
    static *findByLogins({ logins, attributes, transaction }) {
        const includeAuthType = {
            model: AuthType,
            where: { code: this.authType },
            attributes: [],
            as: 'authType'
        };

        return yield User.findAll({
            include: includeAuthType,
            where: { login: logins },
            attributes,
            order: [['id']],
            raw: true,
            transaction
        });
    }

    static _getUserData(webUser, storedUser) {
        return [
            { fieldName: 'login', blackboxName: 'login' },
            { fieldName: 'email', blackboxName: 'address-list.0.address' },
            { fieldName: 'firstname', blackboxName: 'attributes.27' },
            { fieldName: 'lastname', blackboxName: 'attributes.28' }
        ].reduce((result, item) => {
            const storedData = _.get(storedUser, item.fieldName);

            result[item.fieldName] = _.get(webUser, item.blackboxName, storedData);

            return result;
        }, {});
    }

    static _setUserData(userData, yandexUid) {
        _.assign(userData, { yandexUid });
    }

    static get authType() {
        return 'web';
    }

    static getUserInfo(blackboxData) {
        const blackboxUser = _.get(blackboxData, 'users.0');
        const uidValue = _.get(blackboxUser, 'uid.value');

        assert(uidValue, 404, 'User not found', 'UNF');

        return {
            firstname: _.get(blackboxUser, 'attributes.27'),
            lastname: _.get(blackboxUser, 'attributes.28')
        };
    }

    static *getSavedData(uid) {
        const allData = yield WebUser._getAllData(uid);

        if (!allData) {
            return null;
        }

        const { userData, videos } = WebUser._getUserDataAndVideo(allData);
        const { faces, documents } = WebUser._getLinksOfPhotos(allData);

        return { userData, links: { videos, faces, documents } };
    }

    static *_getAllData(uid) {
        const includeCertificate = {
            model: Certificate,
            as: 'certificates',
            attributes: [
                'firstname',
                'lastname',
                'confirmedDate',
                'dueDate',
                'active'
            ]
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['title']
        };
        const includeVideos = {
            model: ProctoringVideos,
            where: { source: 'webcam' },
            as: 'proctoringVideos',
            attributes: ['name'],
            required: false
        };
        const includeTrials = {
            model: Trial,
            as: 'trials',
            attributes: [
                'started',
                'finished',
                'passed',
                'nullified'
            ],
            include: [
                includeCertificate,
                includeTrialTemplate,
                includeVideos
            ]
        };
        const includePhotos = {
            model: UserIdentification,
            as: 'userIdentifications',
            attributes: ['face', 'document']
        };

        return yield User.findOne({
            where: { uid },
            attributes: ['login', 'firstname', 'lastname'],
            include: [includeTrials, includePhotos],
            order: [
                [includeTrials, 'started'],
                [includeTrials, includeVideos, 'startTime'],
                [includePhotos, 'id']
            ]
        });
    }

    static _getUserDataAndVideo(data) {
        const { login, firstname, lastname, trials } = data;
        let videos = [];

        const attempts = trials.map(trial => {
            const trialData = _(trial)
                .pick(['started', 'finished', 'passed', 'nullified'])
                .omitBy(IS_EMPTY_VALUE)
                .value();

            const title = _.get(trial, 'trialTemplate.title');

            trialData.exam = title.replace(SPACE_REG, ' ');

            const certificate = _.get(trial, 'certificates.0');

            if (certificate) {
                trialData.certificate = _(certificate)
                    .pick([
                        'firstname',
                        'lastname',
                        'confirmedDate',
                        'dueDate',
                        'active'
                    ])
                    .omitBy(IS_EMPTY_VALUE)
                    .value();
            }

            const { proctoringVideos } = trial;

            videos = videos.concat(WebUser._getVideosLinks(proctoringVideos));

            return trialData;
        });

        return {
            userData: _.omitBy({ login, firstname, lastname, attempts }, IS_EMPTY_VALUE),
            videos
        };
    }

    static _getVideosLinks(videos) {
        if (_.isEmpty(videos)) {
            return [];
        }

        return videos.map(({ name }) => WebUser._getUrlToFile(name, 'videos'));
    }

    static _getLinksOfPhotos(data) {
        const { userIdentifications } = data;

        return _.transform(userIdentifications, (result, { face, document }) => {
            result.faces.push(WebUser._getUrlToFile(face, ''));
            result.documents.push(WebUser._getUrlToFile(document, ''));
        }, { faces: [], documents: [] });
    }

    static _getUrlToFile(name, directory) {
        const { protocol, host, basePathToFiles } = self;

        return url.format({
            protocol,
            host,
            pathname: path.join(basePathToFiles, directory, name)
        });
    }

    static *saveEmail(userId, email, transaction) {
        yield User.update(
            { email },
            {
                where: { id: userId },
                fields: ['email'],
                transaction
            }
        );
    }
}

module.exports = WebUser;
