'use strict';

const _ = require('lodash');
const assert = require('helpers/assert');
const config = require('yandex-config').s3;
const { UserIdentification } = require('db/postgres');

const Base = require('./base');
const S3 = require('./s3');
const publicS3 = new S3(_.assign({ bucket: config.buckets.public }, config));
const privateS3 = new S3(_.assign({ bucket: config.buckets.private }, config));

const base64ImageRegExp = /^data:image\/(jpe?g|png);base64,/;

function base64ToBuffer(data) {
    assert(data.match(base64ImageRegExp), 400, 'Photo is incorrect', 'PII');

    return Buffer.from(data.replace(base64ImageRegExp, ''), 'base64');
}

class UserIdentificationModel extends Base {
    // eslint-disable-next-line max-params
    static *create(userId, trialTemplateId, document, face) {
        const key = `${userId}_${Date.now()}`;
        const documentKey = `documents/${key}.jpg`;
        const faceKey = `faces/${key}.jpg`;

        yield [
            privateS3.upload(base64ToBuffer(document), documentKey, 'private'),
            publicS3.upload(base64ToBuffer(face), faceKey)
        ];

        yield UserIdentification.create({
            userId,
            trialTemplateId,
            document: documentKey,
            face: faceKey
        });
    }

    static *getLastFace(userId, transaction) {
        const identification = yield UserIdentification.findOne({
            where: { userId },
            order: [['created_at', 'DESC']],
            attributes: ['face'],
            raw: true,
            transaction
        });

        assert(identification, 400, 'Face not found', 'FNF');

        return identification.face;
    }
}

module.exports = UserIdentificationModel;
