'use strict';

const _ = require('lodash');
const assert = require('helpers/assert');
const config = require('yandex-config');
const got = require('got');
const log = require('logger');
const lru = require('lru-cache');
const moment = require('moment');

const cache = lru({ maxAge: config.yt.maxHeavyProxyAge });

class YT {
    static _getMapByCollection(collection) {
        return collection
            .map(JSON.stringify)
            .join('');
    }

    // eslint-disable-next-line complexity
    static *_requestApi(options) {
        let baseUrl = config.yt.host;

        if (options.heavy) {
            const heavyProxy = yield this._getHeavyProxy();

            baseUrl = `http://${heavyProxy}`;
        }

        const mapNode = options.mapNode ? `/${options.mapNode}` : '';
        const tableName = options.tableName ? `/${options.tableName}` : '';
        const pathToTable = `${mapNode}${tableName}`;
        const pathAttributes = options.pathAttributes || '';
        const tableAttributes = options.tableAttributes || '';

        const query = [
            `path=${pathAttributes}${config.yt.path}${pathToTable}`,
            tableAttributes
        ]
            .filter(Boolean)
            .join('&');

        options = _.merge({
            method: 'POST',
            headers: {
                Accept: 'application/json',
                Authorization: `OAuth ${config.yt.authToken}`,
                'Content-Type': 'application/json'
            },
            query,
            path: `${config.yt.api}/${options.command}`
        }, options);

        let response = {};

        try {
            response = yield got(baseUrl, options);
        } catch (err) {
            log.warn(`YT is out of service: ${options.method} ${baseUrl} ${options.body}`, err);
        }

        return response.body;
    }

    static *_getHeavyProxy() {
        if (cache.has('heavyProxy')) {
            return cache.get('heavyProxy');
        }

        let proxies = [];

        try {
            const proxiesData = yield got(`${config.yt.host}/hosts`);

            proxies = JSON.parse(proxiesData.body);
        } catch (err) {
            log.warn('Can not get heavy proxy', err);
        }

        assert(!_.isEmpty(proxies), 500, 'No available YT heavy proxy', 'NAP');

        const [proxy] = proxies;

        cache.set('heavyProxy', proxy);

        return proxy;
    }

    static *createTable(settings) {
        const {
            mapNode,
            tableName,
            schema,
            force = false
        } = settings;

        const options = {
            command: 'create',
            mapNode,
            tableName,
            body: JSON.stringify({
                type: 'table',
                attributes: { schema },
                'output_format': 'json',
                force,
                retry: false,
                trace: false,
                recursive: false,
                'ignore_existing': false
            })
        };

        const body = yield this._requestApi(options);

        assert(body, 500, 'Table not created', 'TNC');

        return body;
    }

    static *_listTables(mapNode, tableAttributes) {
        const options = {
            method: 'GET',
            command: 'list',
            mapNode
        };

        if (tableAttributes) {
            options.tableAttributes = tableAttributes;
        }

        const body = yield this._requestApi(options);

        assert(body, 500, 'Can not list YT tables', 'CLT');

        return JSON.parse(body);
    }

    static *_readTable(mapNode, tableName) {
        const options = {
            method: 'GET',
            command: 'read_table',
            mapNode,
            tableName,
            heavy: true
        };

        const body = yield this._requestApi(options);

        assert(_.isString(body), 500, `Can not read data from YT table: ${tableName}`, 'CRD');

        return {
            name: tableName,
            rows: _(body.split('\n'))
                .compact()
                .map(JSON.parse)
                .value()
        };
    }

    static *_writeTable(data) {
        const { mapNode, tableName, mapping, pathAttributes } = data;
        const encodeUTF = _.defaultTo(data.encodeUTF, true);
        const options = {
            method: 'PUT',
            command: 'write_table',
            heavy: true,
            mapNode,
            tableName,
            headers: {
                'Transfer-Encoding': 'chunked',
                'X-YT-Header-Format': '<format=text>yson',
                'X-YT-Input-Format': `<encode_utf8=${encodeUTF}>json`
            },
            body: mapping,
            pathAttributes
        };

        const body = yield this._requestApi(options);

        assert(!_.isUndefined(body), 500, 'Can not write data to YT', 'CWD');

        return body;
    }

    static *_removeTable(mapNode, tableName) {
        const options = {
            command: 'remove',
            mapNode,
            tableName
        };

        const body = yield this._requestApi(options);

        assert(!_.isUndefined(body), 500, `Can not remove table ${tableName}`, 'CRT');

        return body;
    }

    static *upload(collection) {
        const mapNode = 'input';
        const tableName = `${moment(Date.now()).format('YYYY-MM-DD_HH:mm:ss')}`;
        const mapping = this._getMapByCollection(collection);

        yield this.createTable({
            mapNode,
            tableName,
            schema: config.yt.inputTableSchema
        });

        return yield this._writeTable({ mapNode, tableName, mapping });
    }

    static *loadResults() {
        const mapNode = 'output';
        const tables = yield this._listTables(mapNode);

        return yield tables.map(this._readTable.bind(this, mapNode));
    }

    static *moveToArchive(tables) {
        return yield tables.map(function *(table) {
            const archiveNode = 'archive/output';
            const tableName = table.name;
            const mapping = this._getMapByCollection(table.rows);

            yield this.createTable({
                mapNode: archiveNode,
                tableName
            });
            yield this._writeTable({ mapNode: archiveNode, tableName, mapping });

            return yield this._removeTable('output', tableName);
        }, this);
    }

    static aggregateResults(tables) {
        const intervalsByTrialId = _(tables)
            .map('rows')
            .flatten()
            .groupBy('trialId')
            .reduce((result, intervals, key) => {
                result[key] = {
                    intervals: this._groupByIntervals(intervals),
                    isRevision: intervals[0].isRevision
                };

                return result;
            }, {});

        const trialIds = _.keys(intervalsByTrialId);

        return trialIds.reduce((result, trialId) => {
            const { intervals, isRevision } = intervalsByTrialId[trialId];

            result[trialId] = {
                isViolationsExist: _.some(intervals, interval => interval.hasViolations),
                isRevision,
                intervals: _.values(intervals)
            };

            return result;
        }, {});
    }

    static _groupByIntervals(rows) {
        return _(rows)
            .groupBy(row => {
                return `s${row.start}_e${row.end}`;
            })
            .reduce((result, group) => {
                const [interval] = group;
                const key = `${interval.start}_${interval.end}`;

                result[key] = {
                    hasViolations: this._getAggregatedViolation(group),
                    start: interval.start,
                    end: interval.end,
                    answers: this._getTolokersAnswers(group)
                };

                return result;
            }, {});
    }

    static _getAggregatedViolation(rows) {
        const countedViolations = _.countBy(rows, 'violations');
        const entries = _.entries(countedViolations);
        const violations = _.maxBy(entries, '[1]');

        return violations[0] === 'true';
    }

    static _getTolokersAnswers(rows) {
        return rows.map(row => {
            const {
                violations,
                no_vio_audio_problems: noVioAudioProblems,
                no_vio_no_relate: noVioNoRelate,
                no_vio_other: noVioOther,
                no_vio_other_text: noVioOtherText,
                no_vio_video_problems: noVioVideoProblems,
                vio_cheating: vioCheating,
                vio_diff_user: vioDiffUser,
                vio_other: vioOther,
                vio_other_people: vioOtherPeople,
                vio_other_text: vioOtherText,
                vio_tips: vioTips,
                vio_walk_away_screen: vioWalkAwayScreen
            } = row;

            return {
                violations,
                noVioAudioProblems,
                noVioNoRelate,
                noVioOther,
                noVioOtherText,
                noVioVideoProblems,
                vioCheating,
                vioDiffUser,
                vioOther,
                vioOtherPeople,
                vioOtherText,
                vioTips,
                vioWalkAwayScreen
            };
        });
    }

    static *uploadReports(reports, mapNode = 'reports') {
        const reportNames = _.keys(reports);

        for (const name of reportNames) {
            const mapping = this._getMapByCollection(reports[name]);

            yield this._writeTable({
                mapNode,
                tableName: name,
                mapping,
                encodeUTF: false,
                pathAttributes: '<append=true>'
            });
        }
    }

    static *getTablesByDirs(dirPaths) {
        let allTables = [];
        const tableAttributes = 'attributes[0]=creation_time';

        for (const dirPath of dirPaths) {
            const tables = yield this._listTables(dirPath, tableAttributes);

            const mappedTables = tables.map(table => ({
                dirPath,
                tableName: table.$value,
                creationTime: table.$attributes.creation_time
            }));

            allTables = allTables.concat(mappedTables);
        }

        return allTables;
    }

    static getExpiredTables(tables) {
        const now = moment().startOf('day').toDate();

        return tables.filter(table => {
            const { creationTime } = table;

            const expirationDate = moment(creationTime)
                .add(config.yt.clean.maxTTLMonths, 'month')
                .startOf('day')
                .toDate();

            return expirationDate < now;
        });
    }

    static *removeTables(tables) {
        for (const table of tables) {
            const { dirPath, tableName } = table;

            yield this._removeTable(dirPath, tableName);
        }
    }
}

module.exports = YT;
