const co = require('co');
const fs = require('fs');
const _ = require('lodash');
const config = require('yandex-config').proctoring;

const Excel = require('models/excel');
const ProctorEdu = require('models/proctoring/proctorEdu');

const SIMILAR_BLANK = `templates/reportSimilarProfilesBlank.xlsx`;
const METRICS_BLANK = `templates/reportMetricsBlank.xlsx`;
const METRICS = ['c1', 'c2', 'c3', 'c4', 'b2', 's2'];

const {
    Trial,
    TrialTemplate,
    User
} = require('db/postgres');

function *getUsersWithProctoring() {
    const includeTrialTemplate = {
        model: TrialTemplate,
        as: 'trialTemplate',
        where: { isProctoring: true },
        attributes: []
    };
    const includeTrials = {
        model: Trial,
        as: 'trials',
        attributes: ['id', 'openId'],
        where: {
            started: { $gte: new Date(2019, 5, 24) }
        },
        include: [includeTrialTemplate]
    };

    return yield User.findAll({
        attributes: ['login'],
        include: [includeTrials],
        order: [
            ['login'],
            [includeTrials, 'id']
        ]
    });
}

function *getLastProTrialOpenId(login) {
    const includeTrialTemplate = {
        model: TrialTemplate,
        as: 'trialTemplate',
        where: { isProctoring: true },
        attributes: []
    };
    const includeUser = {
        model: User,
        attributes: [],
        where: { login }
    };
    const lastTrial = yield Trial.findOne({
        attributes: ['openId'],
        include: [includeUser, includeTrialTemplate],
        order: [['started', 'DESC']]
    });

    return _.get(lastTrial, 'openId', '');
}

// eslint-disable-next-line complexity
function *getSimilarUsers(users) {
    const rows = [];

    for (const userData of users) {
        const { login } = userData;
        const proUserData = yield ProctorEdu.getUserData(login);
        const similar = _.get(proUserData, 'similar', []);

        if (!proUserData || _.isEmpty(similar)) {
            rows.push({
                login,
                similarUser: '',
                distance: '',
                similarSession: ''
            });

            continue;
        }

        for (const similarUserData of similar) {
            const { user, distance } = similarUserData;
            const openId = yield getLastProTrialOpenId(user);
            const similarSession = openId ? `${config.host}/#!/top/report?id=${openId}` : '';

            rows.push({
                login,
                similarUser: user,
                distance,
                similarSession
            });
        }
    }

    return rows;
}

function getMetricsValues(sessionData) {
    const intervals = _.get(sessionData, 'timesheet.yaxis', []);

    return METRICS.reduce((res, metric) => {
        res[`avg${metric}`] = _.get(sessionData, ['averages', metric], '');
        res[`max${metric}`] = _.get(_.maxBy(intervals, metric), metric, '');

        return res;
    }, {});
}

// eslint-disable-next-line max-statements
function *getMetricsData(users) {
    const result = [];

    for (const user of users) {
        const { login, trials } = user;

        for (const trial of trials) {
            const { id, openId } = trial;

            console.log('Trial id: ', id);

            const sessionData = yield ProctorEdu.getSessionData(openId);

            // eslint-disable-next-line max-depth
            if (!sessionData) {
                console.log('No session data.');

                continue;
            }

            const metricsData = getMetricsValues(sessionData);
            const values = _.values(metricsData);

            // eslint-disable-next-line max-depth
            if (_.max(values) === 0) {
                console.log('Values are 0.');

                continue;
            }

            const trialRow = _.assign({}, {
                login,
                trialId: id,
                session: `${config.host}/#!/top/report?id=${openId}`
            }, metricsData);

            result.push(trialRow);
        }
    }

    return result;
}

function *getSimilarProfiles(users) {
    const reportData = yield getSimilarUsers(users);

    const blank = fs.readFileSync(SIMILAR_BLANK);
    const excel = Excel.tryLoad(blank);
    const data = excel.write(reportData, false);

    fs.writeFileSync('reportSimilarProfiles.xlsx', data);
}

function *getAverageAndMaxMetrics(users) {
    const reportData = yield getMetricsData(users);

    const blank = fs.readFileSync(METRICS_BLANK);
    const excel = Excel.tryLoad(blank);
    const data = excel.write(reportData, false);

    fs.writeFileSync('reportMetricsValues.xlsx', data);
}

co(function *() {
    const users = yield getUsersWithProctoring();

    yield getSimilarProfiles(users);
    yield getAverageAndMaxMetrics(users);

    console.log('Success process');
});
