const parse = require('co-busboy');
const getRawBody = require('raw-body');
const _ = require('lodash');

const { fullFilledReports } = require('yandex-config');

const assert = require('helpers/assert');
const assertIsNumber = assert.isNumber;
const assertIsValidEmail = assert.isValidEmail;

const AccessControl = require('accessControl/admin');

const AdminUser = require('models/adminUser');
const AgenciesLoader = require('models/agenciesLoader');
const Attempt = require('models/attempt');
const Authority = require('models/authority');
const Certificate = require('models/certificate');
const Direct = require('models/direct');
const Draft = require('models/draft');
const Exam = require('models/exam');
const Excel = require('models/excel');
const FreezingModel = require('models/freezing');
const Loader = require('models/loader');
const LockModel = require('models/lock');
const Notifier = require('models/notifier');
const ProctoringResponse = require('models/proctoringResponse');
const ProctoringVideos = require('models/proctoringVideos');
const Report = require('models/report');
const S3 = require('models/s3');
const GlobalUser = require('models/globalUser');
const WebUser = require('models/user/webUser');

module.exports = {
    *getAdminData() {
        const accessControl = new AccessControl(this.state);

        accessControl.authorizationRequired();

        const authority = yield Authority.find(accessControl.uid);

        accessControl.hasAnyAccess(authority);

        const lastSyncAgenciesDate = yield Direct.getLastSyncDate();
        const currentFrozenExams = yield FreezingModel.getCurrentFrozenExams();
        const lockedExams = yield LockModel.findLockedExams();

        const access = authority.getRoles();

        this.body = { lastSyncAgenciesDate, currentFrozenExams, lockedExams, access };
    },

    *downloadXLSX() {
        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAccessToEditExam();

        const { examId } = this.params;

        assertIsNumber(examId, 400, 'Exam id is invalid', 'EII', { examId });

        const loader = new Loader();

        this.state.blank = 'blank.xlsx';
        this.body = yield loader.select(examId);
    },

    *downloadJSON() {
        const { transaction } = this.state;

        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAccessToEditExam();

        const exam = yield Exam.findByIdentity(this.params.examIdentity, transaction);

        this.body = yield Exam.getExamData(exam.id, transaction);
    },

    *parseXLSX() {
        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAccessToEditExam();

        const file = yield parse(this);
        const content = yield getRawBody(file);
        const excel = Excel.tryLoad(content);

        this.body = yield excel.getOperations();
    },

    *loadData() {
        const { transaction } = this.state;

        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAdminAccess(transaction);

        const { draft, examId } = this.request.body || {};

        assertIsNumber(examId, 400, 'Exam id is invalid', 'EII', { examId });
        assert.bySchema(draft, 'Draft');

        const admin = yield AdminUser.findOrCreate(accessControl.login, this.ip, transaction);

        const savedDraft = yield Draft.create({
            exam: draft,
            adminId: admin.id,
            trialTemplateId: examId
        }, transaction);

        const loader = new Loader(draft);

        loader.checkExamId(examId);
        yield loader.upsert(transaction);

        yield Draft.updateStatus(savedDraft.id, 'published', transaction);

        this.status = 204;
    },

    *syncAgenciesInfo() {
        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAnalystAccess();

        yield Direct.syncAgenciesInfo();

        this.status = 204;
    },

    *getAgenciesInfo() {
        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAnalystAccess();

        this.state.blank = 'agenciesBlank.xlsx';
        this.state.isFullFilling = true;
        this.body = yield AgenciesLoader.getFlatData();
    },

    *freeze() {
        const accessControl = new AccessControl(this.state);

        yield accessControl.hasDeveloperAccess();

        const examIds = _.get(this.request, 'body.examIds');

        assert(_.isArray(examIds), 400, 'examIds should be array', 'ESA', { examIds });
        examIds.forEach(examId =>
            assert.isNumber(examId, 400, 'Exam identity is invalid', 'EII', { examId })
        );

        const finishTime = yield FreezingModel.freeze(
            accessControl.uid,
            examIds,
            this.state.transaction
        );

        const examsInfo = yield Exam.getInfoByIds(examIds);

        yield Notifier.freezeNotify(accessControl.login, finishTime, examsInfo);

        const lastAttemptFinish = yield FreezingModel.getLastAttemptFinish(examIds);

        this.body = { lastAttemptFinish };
    },

    *unfreeze() {
        const accessControl = new AccessControl(this.state);

        yield accessControl.hasDeveloperAccess();

        const examIds = _.get(this.request, 'body.examIds');

        assert(_.isArray(examIds), 400, 'examIds should be array', 'ESA', { examIds });
        examIds.forEach(examId =>
            assert.isNumber(examId, 400, 'Exam identity is invalid', 'EII', { examId })
        );

        yield FreezingModel.unfreeze(accessControl.uid, examIds, this.state.transaction);

        const examsInfo = yield Exam.getInfoByIds(examIds);

        yield Notifier.unfreezeNotify(accessControl.login, examsInfo);

        this.status = 204;
    },

    *report() {
        const accessControl = new AccessControl(this.state);

        accessControl.authorizationRequired();

        const authority = yield Authority.find(accessControl.uid);
        const userRoles = authority.getRoles();

        const { reportType } = this.params;
        const report = new Report(reportType, this.query);
        const { data, blankName } = yield report.getReportData(userRoles);

        this.state.isFullFilling = _.includes(fullFilledReports, reportType);
        this.state.blank = this.query.format === 'xlsx' ? `${blankName}Blank.xlsx` : '';
        this.body = data;
    },

    *reports() {
        const accessControl = new AccessControl(this.state);

        accessControl.authorizationRequired();

        const authority = yield Authority.find(accessControl.uid);
        const userRoles = authority.getRoles();

        this.body = Report.getReports(userRoles);
    },

    *getVideosData() {
        const { attemptId } = this.params;

        assertIsNumber(attemptId, 400, 'Attempt id is invalid', 'AII', { attemptId });

        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAccessToVideo();

        const videosNames = yield ProctoringVideos.getVideosNames(attemptId, ['webcam', 'screen']);

        const attempt = yield Attempt.findById(attemptId);
        const appealData = yield Attempt.tryFindCertAndAppeal(attemptId);
        const additionalInfo = yield ProctoringResponse.getAdditionalInfo(attemptId);

        this.body = {
            videos: _.map(videosNames, name => S3.getPathToProxyS3('public', name, 'videos')),
            pdfLink: S3.getPathToProxyS3('public', attempt.pdf, 'pdf'),
            appealData,
            additionalInfo
        };
    },

    *retryProctoring() {
        const { transaction } = this.state;
        const { attemptId } = this.params;

        assertIsNumber(attemptId, 400, 'Attempt id is invalid', 'AII', { attemptId });

        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAdminAccess();

        const attempt = yield Attempt.findById(attemptId);
        const proctoringAnswer = yield attempt.processProctoringAnswer(false, transaction);
        const isRetryFailed = proctoringAnswer.source === 'proctoring' && _.isNull(proctoringAnswer.evaluation);

        assert(!isRetryFailed, 424, 'Proctoring not send evaluation', 'PNE');

        yield Certificate.tryCreate(attempt, {
            isProctoring: true, isProctoringCorrect: proctoringAnswer.isCorrect()
        }, transaction);

        this.body = {
            passed: attempt.get('passed'),
            verdict: proctoringAnswer.verdict
        };
    },

    // eslint-disable-next-line max-statements
    *appeal() {
        const { transaction } = this.state;

        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAccessToVideo(transaction);

        const { attemptId, verdict, email } = this.request.body;

        assertIsNumber(attemptId, 400, 'Attempt id is invalid', 'AII', { attemptId });
        assert(_.includes(['correct', 'failed'], verdict), 400, 'Verdict is invalid', 'VII', { verdict });
        assertIsValidEmail(email, 400, 'User email is invalid', 'UEI', { email });

        const lastResponse = yield ProctoringResponse.tryFindLast(attemptId, transaction);

        assert(!lastResponse || lastResponse.source !== 'appeal', 403, 'The appeal has already been made', 'AAM');

        const attempt = yield Attempt.findById(attemptId, transaction);
        const isPassed = attempt.get('passed');

        const dataForAppeal = [{
            source: 'appeal',
            verdict
        }];

        yield ProctoringResponse.create(attemptId, dataForAppeal, transaction);

        if (isPassed && verdict === 'correct') {
            yield Certificate.create({ id: attemptId }, transaction);
        }

        yield Notifier.sendLetterToUser(email, verdict);

        this.status = 204;
    },

    *sendDraftForModeration() {
        const { transaction } = this.state;

        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAccessToEditExam(transaction);

        const { draft, examIdentity } = this.request.body || {};
        const exam = yield Exam.findByIdentity(examIdentity, transaction);

        assert.bySchema(draft, 'Draft');

        const admin = yield AdminUser.findOrCreate(accessControl.login, this.ip, transaction);

        const savedDraft = yield Draft.create({
            exam: draft,
            adminId: admin.id,
            trialTemplateId: exam.id
        }, transaction);

        yield Draft.updateStatus(savedDraft.id, 'on_moderation', transaction);

        yield Notifier.draftForModerationNotify(examIdentity, accessControl.login);

        this.status = 204;
    },

    *getUserAssociations() {
        const accessControl = new AccessControl(this.state);

        yield accessControl.hasSupportAccess();

        const { login } = this.query || {};

        assert(_.isString(login), 400, 'Login should be a string', 'LSS', { login });

        const user = yield WebUser.findUser({ where: { login }, attributes: ['globalUserId'] });

        assert(user, 404, 'User not found', 'UNF', { login });

        const { globalUserId } = user;

        if (!globalUserId) {
            this.body = { logins: [], globalUserInfo: {} };

            return;
        }

        const [associatedUsers, globalUserInfo] = yield [
            WebUser.getAssociatedUsers([globalUserId]),
            GlobalUser.getBansInfo(globalUserId)
        ];

        this.body = {
            logins: _.map(associatedUsers, 'login'),
            globalUserInfo
        };
    },

    nullifyAttempts: function *nullify() {
        const accessControl = new AccessControl(this.state);

        yield accessControl.hasSupportAccess();

        const { attemptIds } = this.request.body || {};
        const { transaction } = this.state;

        assert(_.isArray(attemptIds), 400, 'attemptIds should be array', 'ASA', { attemptIds });
        attemptIds.forEach(attemptId => assert.isNumber(attemptId, 400, 'Attempt id is invalid', 'AII', { attemptId }));

        const storedAttempts = yield Attempt.findByIds(attemptIds, ['id'], transaction);
        const storedAttemptsIds = _.map(storedAttempts, 'id');
        const notFoundAttempts = _.difference(attemptIds, storedAttemptsIds);

        yield Attempt.nullify(attemptIds, 'manual', transaction);

        this.body = { notFoundAttempts };
    }
};
