const _ = require('lodash');
const config = require('yandex-config');

const AttemptAccessControl = require('accessControl/attempt');
const AdminAccessControl = require('accessControl/admin');

const Attempt = require('models/attempt');
const AttemptQuestion = require('models/attemptQuestion');
const Certificate = require('models/certificate');
const ProctoringResponse = require('models/proctoringResponse');
const WebUser = require('models/user/webUser');

const assert = require('helpers/assert');
const hasher = require('helpers/idHasher');
const parseFormFields = require('helpers/parseFormFields');

function *checkNeedNullify(attempt, { requestNullify, isCritMetrics }) {
    const userId = attempt.get('userId');
    const trialTemplateId = attempt.get('trialTemplateId');

    if (isCritMetrics) {
        const isLimitExceeded = yield Attempt.isNullifiesByMetricsLimitExceeded(
            userId,
            trialTemplateId
        );

        return requestNullify && !isLimitExceeded;
    }

    return requestNullify;
}

function *trySendCertificateToGeoadv(attemptId, uid, transaction) {
    const { exams } = config.geoadv;

    const attemptInfo = yield Attempt.getAttemptInfo(attemptId, transaction);
    const certId = _.get(attemptInfo, 'certificates.0.id');
    const examSlug = _.get(attemptInfo, 'trialTemplate.slug');

    if (!exams.includes(examSlug) || !certId) {
        return;
    }

    yield Certificate.sendCertificatesToGeoadv([{ certId, uid, examSlug }], transaction);
}

module.exports = {
    *create() {
        const { state, params: { examIdentity } } = this;

        const openId = _.get(this.request, 'body.openId');

        if (openId) {
            assert.isOpenId(openId, 400, 'openId is invalid', 'OII', { openId });
        }

        const accessControl = new AttemptAccessControl(state);

        yield accessControl.hasAccessToCreateAttempt(examIdentity, state.transaction);

        const dataToCreate = { examIdentity, openId };
        const attempt = yield Attempt.create(dataToCreate, state, accessControl.authType);

        this.body = attempt.toJSON();
    },

    *question() {
        const { attemptId, questionNumber } = this.params;

        assert.isNumber(attemptId, 400, 'Attempt id is invalid', 'AII', { attemptId });
        assert.isNumber(
            questionNumber,
            400,
            'Question number is invalid',
            'QNI',
            { questionNumber }
        );

        const attempt = yield Attempt.findById(attemptId);

        const accessControl = new AttemptAccessControl(this.state);

        yield accessControl.hasAccessToQuestion(attempt);

        const attemptQuestion = yield AttemptQuestion.findByNumber(attemptId, questionNumber);

        this.body = attemptQuestion.toJSON();
    },

    *nextQuestion() {
        const { attemptId } = this.params;

        assert.isNumber(attemptId, 400, 'Attempt id is invalid', 'AII', { attemptId });

        const attempt = yield Attempt.findById(attemptId);

        const accessControl = new AttemptAccessControl(this.state);

        yield accessControl.hasAccessToQuestion(attempt);

        const attemptQuestion = yield AttemptQuestion.findNext(attemptId);

        this.body = attemptQuestion.toJSON();
    },

    *check() {
        const { examIdentity } = this.params;

        const accessControl = new AttemptAccessControl(this.state);

        const attemptInfo = yield Attempt.getInfo(
            examIdentity,
            {
                uid: accessControl.uid,
                authTypeCode: accessControl.authType,
                login: accessControl.login
            }
        );

        this.body = _.assign(attemptInfo, {
            firstname: _.get(this.state, 'user.attributes.27'),
            lastname: _.get(this.state, 'user.attributes.28')
        });
    },

    *answer() {
        const { attemptId, seq: questionNumber, answerId } = this.request.body || {};

        assert.isNumber(attemptId, 400, 'Attempt id is invalid', 'AII', { attemptId });
        assert.isNumber(
            questionNumber,
            400,
            'Question number is invalid',
            'QNI',
            { questionNumber }
        );

        const attempt = yield Attempt.findById(attemptId);

        const questionAccessControl = new AttemptAccessControl(this.state);

        yield questionAccessControl.hasAccessToQuestion(attempt);

        const question = yield AttemptQuestion.findByNumber(attemptId, questionNumber);

        yield question.answer(answerId);

        this.status = 204;
    },

    // eslint-disable-next-line max-statements
    *finish() {
        const { transaction } = this.state;
        const { attemptId } = this.params;
        const isCritMetrics = _.get(this.request, 'body.isCritMetrics', false);
        const requestNullify = _.get(this.request, 'body.requestNullify', false);

        assert.isNumber(attemptId, 400, 'Attempt id is invalid', 'AII', { attemptId });
        assert.isBoolean(isCritMetrics, 400, 'Crit metrics param is invalid', 'CMI', { isCritMetrics });
        assert.isBoolean(requestNullify, 400, 'Need nullify param is invalid', 'NNI', { requestNullify });

        const attempt = yield Attempt.findById(attemptId, transaction);

        const accessControl = new AttemptAccessControl(this.state);

        yield accessControl.hasAccessToFinish(attempt, transaction);

        yield attempt.finish(isCritMetrics, transaction);

        const needNullify = yield checkNeedNullify(attempt, { isCritMetrics, requestNullify });

        if (needNullify) {
            const nullifyReason = isCritMetrics ? 'metrics' : 'manual';

            yield Attempt.nullify([attemptId], nullifyReason, transaction);
        }

        if (!needNullify) {
            yield trySendCertificateToGeoadv(attemptId, accessControl.uid, transaction);
        }

        this.body = { isNullified: needNullify };
    },

    *result() {
        const { attemptId } = this.params;
        const { transaction } = this.state;

        assert.isNumber(attemptId, 400, 'Attempt id is invalid', 'AII', { attemptId });

        const attempt = yield Attempt.findById(attemptId, transaction);

        const accessControl = new AttemptAccessControl(this.state);

        yield accessControl.hasAccessToAttempt(attempt, transaction);

        const body = yield attempt.getResult(
            {
                uid: accessControl.uid,
                authType: accessControl.authType,
                login: accessControl.login
            },
            transaction
        );

        body.hashedUserId = hasher.encodeUserId(accessControl.uid);
        this.body = body;
    },

    *attemptsInfo() {
        const accessControl = new AdminAccessControl(this.state);

        yield accessControl.hasSupportAccess();

        const { attemptIds } = this.request.body || {};

        assert(_.isArray(attemptIds), 400, 'attemptIds should be array', 'ASA', { attemptIds });
        attemptIds.forEach(attemptId =>
            assert.isNumber(attemptId, 400, 'Attempt id is invalid', 'AII', { attemptId })
        );

        const attemptsInfo = yield Attempt.getTrialsInfo(attemptIds);

        this.body = { attemptsInfo };
    },

    *needRevision() {
        const { attemptId } = this.params;
        const { transaction } = this.state;
        const { email } = parseFormFields(this.request.body || {});

        assert.isNumber(attemptId, 400, 'Attempt id is invalid', 'AII', { attemptId });
        assert.isValidEmail(email, 400, 'User email is invalid', 'UEI', { email });

        const [attemptData] = yield Attempt.findByIds(
            [attemptId],
            ['userId', 'passed'],
            transaction
        );

        assert(attemptData, 404, 'Attempt not found', 'ATF', { attemptId });

        const responses = yield ProctoringResponse.findByTrialIds([attemptId], transaction);

        AttemptAccessControl.hasAccessForRevision(responses, attemptId, attemptData.passed);

        yield ProctoringResponse.setIsRevisionRequested(attemptId, transaction);
        yield WebUser.saveEmail(attemptData.userId, email, transaction);

        this.status = 200;
    },

    *saveProctoringMetrics() {
        const { attemptId } = this.params;
        const { transaction } = this.state;
        const proctoringMetrics = this.request.body;

        assert.isNumber(attemptId, 400, 'Attempt id is invalid', 'AII', { attemptId });
        assert.bySchema(proctoringMetrics, 'ProctoringMetrics');

        const attempt = yield Attempt.findById(attemptId);

        const attemptAccessControl = new AttemptAccessControl(this.state);

        yield attemptAccessControl.hasAccessToAttempt(attempt);

        yield Attempt.saveProctoringMetrics(attemptId, proctoringMetrics, transaction);

        this.status = 204;
    },

    *getProctoringMetrics() {
        const { attemptId } = this.params;

        assert.isNumber(attemptId, 400, 'Attempt id is invalid', 'AII', { attemptId });

        const attempt = yield Attempt.findById(attemptId);

        const attemptAccessControl = new AttemptAccessControl(this.state);

        yield attemptAccessControl.hasAccessToAttempt(attempt);

        const proctoringMetrics = yield Attempt.getProctoringMetrics(attemptId);

        this.body = proctoringMetrics || {};
    }
};
