const AdminAccessControl = require('accessControl/admin');
const CertificateAccessControl = require('accessControl/certificate');

const CertificateModel = require('models/certificate');
const WebUser = require('models/user/webUser');
const AttemptModel = require('models/attempt');
const Blackbox = require('models/blackbox');

const config = require('yandex-config');
const _ = require('lodash');

const assert = require('helpers/assert');
const hasher = require('helpers/idHasher');
const sendLetters = require('helpers/sendLettersForNullifyCert');

module.exports = {
    *find() {
        const body = _.get(this.request, 'body', {});
        const { certId, lastname } = body;

        const accessControl = new CertificateAccessControl(this.state);

        accessControl.hasAccessForFindCertificate(certId, lastname);

        const certificate = yield CertificateModel.find(certId, lastname);

        this.body = yield CertificateModel.pickCertificateData(certificate);
    },

    *findAll() {
        const { hashedUserId } = this.params;

        const accessControl = new CertificateAccessControl(this.state);

        accessControl.hasAccessForFindAllCertificates(hashedUserId);

        const uid = hasher.decodeUserId(hashedUserId);
        const data = yield CertificateModel.findAll(uid);

        if (data.firstname || data.lastname) {
            this.body = data;
        } else {
            const blackbox = new Blackbox(config.blackbox);
            const userip = this.header.userip || this.ip; // FIXME: EXPERTDEV-745
            const blackboxData = yield blackbox.userInfo({ uid, userip });
            const userInfo = WebUser.getUserInfo(blackboxData);

            this.body = _(userInfo).assign({ certificates: data.certificates });
        }
    },

    *myCertificates() {
        let { exams } = this.query;

        if (!Array.isArray(exams)) {
            exams = [exams];
        }

        const accessControl = new CertificateAccessControl(this.state);

        accessControl.hasAccessForGetUserCertificates(exams);

        const getAttemptInfo = AttemptModel.getInfo.bind(AttemptModel);
        const { uid, authType: authTypeCode, login } = accessControl;
        const userData = { uid, login, authTypeCode };

        this.body = yield {
            hashedUserId: hasher.encodeUserId(uid),
            certificates: CertificateModel.getMyCertificates(uid, login, getAttemptInfo),
            failedTrials: AttemptModel.getLastFailedTrials(exams, userData),
            pendingTrials: AttemptModel.getPendingTrials(exams, userData),
            nullifiedCerts: CertificateModel.getNullifiedCertificates(uid)
        };
    },

    *getPdf() {
        const { hashedUserId, certId } = this.params;

        assert.isNumber(certId, 400, 'Certificate id is invalid', 'CII', { certId });

        const accessControl = new CertificateAccessControl(this.state);

        accessControl.hasAccessForFindAllCertificates(hashedUserId);

        const uid = hasher.decodeUserId(hashedUserId);
        const pdf = yield CertificateModel.getPdf(certId, uid);

        this.type = 'application/pdf';
        this.body = pdf;
    },

    *findByUids() {
        const uids = _.get(this.request, 'body.uids');

        assert(_.isArray(uids), 400, 'Uids is required and should be array', 'USA');
        uids.forEach(uid => assert.isNumber(uid, 400, 'Uid is invalid', 'UII', { uid }));

        this.body = yield CertificateModel.findCertsByUids(uids);
    },

    *nullify() {
        const { transaction } = this.state;

        const accessControl = new AdminAccessControl(this.state);

        yield accessControl.hasSupportAccess();

        const certIds = _.get(this.request, 'body.certIds');

        assert(_.isArray(certIds), 400, 'certIds is required and should be array', 'CSA', { certIds });
        certIds.forEach(certId => assert.isNumber(certId, 400, 'Certificate id is invalid', 'CII', { certId }));

        yield CertificateModel.nullify(certIds, 'rules', transaction);

        const usersData = yield CertificateModel.getUsersForNullifyCerts(certIds, transaction);

        yield sendLetters(usersData, 'nullifyCert');

        this.status = 204;
    },

    *certificatesInfo() {
        const accessControl = new AdminAccessControl(this.state);

        yield accessControl.hasSupportAccess();

        const { certIds } = this.request.body || {};

        assert(_.isArray(certIds), 400, 'certIds is required and should be array', 'CSA', { certIds });
        certIds.forEach(certId => assert.isNumber(certId, 400, 'Certificate id is invalid', 'CII', { certId }));

        const certificatesInfo = yield CertificateModel.getCertificatesInfo(certIds);

        this.body = { certificatesInfo };
    },

    *sendToGeoadv() {
        const certsData = yield CertificateModel.getGeoadvCertificates();

        yield CertificateModel.sendCertificatesToGeoadv(certsData);

        this.status = 204;
    }
};
