const _ = require('lodash');

const assert = require('helpers/assert');
const assertIsNumber = assert.isNumber;

const AccessControl = require('accessControl/admin');
const AdminUser = require('models/adminUser');
const Draft = require('models/draft');
const Exam = require('models/exam');
const LockModel = require('models/lock');

module.exports = {
    *find() {
        const exam = yield Exam.findByIdentity(this.params.identity);

        this.body = exam.toJSON();
    },

    *findByCluster() {
        const clusterSlug = this.params.slug;

        assert.isSlug(clusterSlug, 400, 'Cluster slug is invalid', 'CSI', { clusterSlug });

        const exams = yield Exam.findByCluster(clusterSlug);

        this.body = exams;
    },

    *updateSettings() {
        const { transaction } = this.state;

        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAdminAccess(transaction);

        const exam = yield Exam.findByIdentity(this.params.identity, transaction);

        const { body } = this.request;
        const { timeLimit } = body || {};

        assert(_.isNumber(timeLimit) && timeLimit > 0, 400, 'Time limit is invalid', 'TLI', { timeLimit });

        [
            'title',
            'description',
            'rules',
            'seoDescription',
            'ogDescription'
        ].forEach(field => {
            assert(_.isString(body[field]), 400, 'Field is not a string', 'FNS', { [field]: body[field] });
        });

        yield exam.updateSettings(body, transaction);

        this.body = exam.toJSON();
    },

    *saveDraft() {
        const { transaction } = this.state;

        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAccessToEditExam(transaction);

        const exam = yield Exam.findByIdentity(this.params.identity, transaction);

        const { draft } = this.request.body || {};

        assert.bySchema(draft, 'Draft');

        const admin = yield AdminUser.findOrCreate(accessControl.login, this.ip, transaction);

        const data = {
            trialTemplateId: exam.id,
            adminId: admin.id,
            exam: draft
        };

        yield Draft.create(data, transaction);

        this.status = 204;
    },

    *findDraft() {
        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAccessToEditExam();

        const examId = this.params.identity;

        assertIsNumber(examId, 400, 'Exam id is invalid', 'EII', { examId });

        this.body = yield Draft.find(examId);
    },

    *lock() {
        const { transaction } = this.state;

        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAccessToEditExam(transaction);

        const exam = yield Exam.findByIdentity(this.params.identity, transaction);

        const unlockDate = yield accessControl.hasAccessToLockExam(exam.id, transaction);

        if (!_.isNull(unlockDate)) {
            const admin = yield AdminUser.findOrCreate(accessControl.login, this.ip, transaction);

            yield LockModel.lock({ adminId: admin.id, trialTemplateId: exam.id }, transaction);
        }

        this.status = 204;
    },

    *unlock() {
        const { transaction } = this.state;

        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAccessToEditExam(transaction);

        const exam = yield Exam.findByIdentity(this.params.identity, transaction);
        const lock = yield LockModel.tryFindLast(exam.id, transaction);

        assert(lock && !lock.unlockDate, 400, 'Exam is not locked', 'ENL', { examId: exam.id });

        yield LockModel.unlock(lock.id, transaction);

        this.status = 204;
    },

    *getLastVersion() {
        const { transaction } = this.state;

        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAccessToEditExam(transaction);

        const exam = yield Exam.findByIdentity(this.params.identity, transaction);

        const draft = yield Draft.find(exam.id, transaction);
        const isUnpublishedDraft = draft &&
            draft.status !== 'published' &&
            draft.status !== 'ignored';

        this.body = isUnpublishedDraft ?
            draft.exam :
            yield Exam.getExamData(exam.id, transaction);
    },

    *ignoreDraft() {
        const { transaction } = this.state;

        const accessControl = new AccessControl(this.state);

        yield accessControl.hasAccessToEditExam(transaction);

        const exam = yield Exam.findByIdentity(this.params.identity, transaction);
        const draft = yield Draft.find(exam.id, transaction);

        assert(draft, 404, 'Draft not found', 'DNF', { examId: exam.id });

        const admin = yield AdminUser.findOrCreate(accessControl.login, this.ip, transaction);

        const dataForCreate = {
            trialTemplateId: exam.id,
            adminId: admin.id,
            exam: draft.exam,
            status: 'ignored'
        };

        yield Draft.create(dataForCreate, transaction);

        this.status = 204;
    }
};
