'use strict';

const _ = require('lodash');
const assert = require('helpers/assert');
const fs = require('fs');
const log = require('logger');
const s3Config = require('yandex-config').s3;

const { removeFiles } = require('helpers/files');

const ProctorEdu = require('models/proctoring/proctorEdu');
const AttemptModel = require('models/attempt');
const ProctoringVideos = require('models/proctoringVideos');
const S3 = require('models/s3');

const s3 = new S3(_.assign({ bucket: s3Config.buckets.public }, s3Config));

const { sequelize } = require('db/postgres');
const co = require('co');

function *saveVideosAndPdfInDB(data, transaction) {
    const {
        trialId,
        videosData,
        openId,
        pdfName: pdf
    } = data;

    yield ProctoringVideos.create(trialId, videosData, transaction);
    yield AttemptModel.setFilesStatusAndPdf(openId, { filesStatus: 'saved', pdf }, transaction);
}

function *saveDataToS3({ videosData, pdfName, pdf }) {
    for (const videoData of videosData) {
        const { name, videoPath } = videoData;

        const video = fs.readFileSync(videoPath);

        yield s3.upload(Buffer.from(video), `videos/${name}`);
    }

    yield s3.upload(Buffer.from(pdf), `pdf/${pdfName}`);
}

function *trySaveToS3AndRemoveLocaFiles(data) {
    let error = null;

    try {
        yield saveDataToS3(data);
    } catch (err) {
        error = err;
    }

    const videoPaths = _.map(data.videosData, 'videoPath');

    removeFiles(videoPaths);

    if (error) {
        throw error;
    }
}

function *setFilesStatusToLoading(trials, transaction) {
    for (const trial of trials) {
        yield AttemptModel.setFilesStatusAndPdf(trial.openId, { filesStatus: 'loading' }, transaction);
    }
}

function _skipIncompleteTrial(sessionData) {
    return !sessionData || !sessionData.complete || sessionData.status !== 'stopped';
}

function _filesStatusForSkippedTrial(sessionData) {
    return !sessionData || !(sessionData.complete || sessionData.status === 'created') ? 'initial' : 'error';
}

function _logIncompleteTrial(sessionData) {
    return sessionData && sessionData.status === 'created';
}

// eslint-disable-next-line max-statements
function *saveTrialData(trialId, openId) {
    log.info('Loading proctoring data for session', { openId });
    const sessionData = yield ProctorEdu.getSessionData(openId);
    const toSkip = {
        skip: _skipIncompleteTrial(sessionData),
        filesStatus: _filesStatusForSkippedTrial(sessionData),
        toLog: _logIncompleteTrial(sessionData)
    };

    if (toSkip.skip) {
        log.info('Session data is incomplete', { toSkip });
        yield AttemptModel.setFilesStatusAndPdf(openId, _.pick(toSkip, ['filesStatus']));

        if (toSkip.toLog) {
            log.error(`Incomplete but expired trial with proctoring status 'created'`,
                { trialId, openId, sessionData });
        }

        return;
    }

    const videosData = yield ProctorEdu.getConcatenatedVideos(openId, trialId);

    const sessionPdf = _.get(sessionData, 'pdf');

    log.info('Loading session pdf', { sessionPdf });
    const pdf = yield ProctorEdu.getFile(sessionPdf, 'pdf');

    assert(pdf, 424, 'ProctorEdu pdf not loaded', 'PNL', { openId, sessionPdf });

    const pdfName = `${sessionPdf}.pdf`;

    log.info('Trying to save session files to S3', { dataToSave: { videosData, pdf, pdfName } });
    yield trySaveToS3AndRemoveLocaFiles({ videosData, pdf, pdfName });

    const dataToSave = { trialId, openId, videosData, pdfName };

    yield sequelize.transaction(co.wrap(saveVideosAndPdfInDB.bind(null, dataToSave)));
}

function *uploadFilesAsync() {
    const trials = yield AttemptModel.getTrialsWithInitialFilesStatus();

    yield sequelize.transaction(co.wrap(setFilesStatusToLoading.bind(null, trials)));

    for (const trial of trials) {
        const { id: trialId, openId } = trial;

        try {
            yield saveTrialData(trialId, openId);
        } catch (error) {
            yield AttemptModel.setFilesStatusAndPdf(openId, { filesStatus: 'error' });

            log.error('Upload data was failed', { error });
        }
    }
}

module.exports = {
    uploadFilesAsync,

    // eslint-disable-next-line require-yield
    *uploadFiles() {
        co(uploadFilesAsync);

        this.status = 204;
    }
};
