const _ = require('lodash');
const got = require('got');

const assert = require('helpers/assert');
const sendNullifyLetters = require('helpers/sendLettersForNullifyCert');

const AdminAccessControl = require('accessControl/admin');
const AttemptAccessControl = require('accessControl/attempt');
const AdminUser = require('models/adminUser');
const Certificate = require('models/certificate');
const GlobalUser = require('models/globalUser');
const Ban = require('models/ban');
const Exam = require('models/exam');
const S3Model = require('models/s3');
const UserIdentification = require('models/userIdentification');
const UserModel = require('models/user');
const WebUserModel = require('models/user/webUser');

function *getExamId(examIdentity) {
    assert.isSlug(examIdentity, 400, 'Exam identity is incorrect', 'EII', { examIdentity });

    if (/^\d+$/.test(examIdentity)) {
        return examIdentity;
    }

    const exam = yield Exam.findByIdentity(examIdentity);

    return exam.id;
}

module.exports = {
    *createPhoto() {
        const accessControl = new AttemptAccessControl(this.state);
        const examIdentity = yield getExamId(this.params.examIdentity);

        yield accessControl.hasAccessToCreateAttempt(examIdentity);

        const { document, face } = _.get(this.request, 'body', {});

        assert(document, 400, 'Document photo is absent', 'PIA');
        assert(face, 400, 'Face photo is absent', 'PIA');

        const User = new UserModel(accessControl.authType);
        const user = yield User.findAndRenew(this.state.user);

        yield UserIdentification.create(user.id, examIdentity, document, face);

        this.status = 201;
    },

    *getUserData() {
        const { uid } = this.request.body || {};
        const { tvmClient } = this.state;

        assert(tvmClient === 'takeout', 403, 'Client has no access', 'CNA', { tvmClient });
        assert.isNumber(uid, 400, 'Uid is invalid', 'UII', { uid });

        const data = yield WebUserModel.getSavedData(uid);

        if (!data) {
            this.body = { status: 'no_data' };

            return;
        }

        const { links: { videos, faces, documents } } = data;
        const fileLinks = videos.concat(faces, documents);

        const result = { status: 'ok', data: { 'user.json': JSON.stringify(data.userData) } };

        if (!_.isEmpty(fileLinks)) {
            // eslint-disable-next-line camelcase
            result.file_links = fileLinks;
        }

        this.body = result;
    },

    // eslint-disable-next-line require-yield
    *getUserFile() {
        const { tvmClient } = this.state;
        const { dir, name } = this.params;

        assert(tvmClient === 'takeout', 403, 'Client has no access', 'CNA', { tvmClient });
        assert(_.includes(['videos', 'faces', 'documents'], dir), 400, 'Directory is invalid', 'DII', { dir });
        assert(/^[/a-z0-9_-]+.(jpg|webm)$/.test(name), 400, 'File name is invalid', 'FNI', { name });

        const fileUrl = S3Model.getPathToS3Read(dir, name);
        const fileName = name.replace(/\w+_/, '');

        this.set({ 'Content-Disposition': `attachment; filename="${fileName}"` });
        this.body = got.stream(fileUrl);
    },

    *ban() {
        const accessControl = new AdminAccessControl(this.state);

        yield accessControl.hasSupportAccess();

        assert.bySchema(this.request.body, 'Ban');

        const { login, isSuperban, reason, trialTemplateIds } = this.request.body;
        const { transaction } = this.state;

        const storedUser = yield WebUserModel.findUser({ where: { login }, transaction });

        assert(storedUser, 404, 'User not found', 'UNF', { login });

        const globalUserId = yield GlobalUser.getOrCreate(storedUser, transaction);
        const admin = yield AdminUser.findOrCreate(accessControl.login, this.ip, transaction);

        if (isSuperban) {
            yield GlobalUser.updateSuperban({ globalUserId, isBanned: true }, transaction);
        }

        yield Ban.banUser({
            globalUserId,
            adminId: admin.id,
            reason,
            trialTemplateIds,
            isSuperban,
            userLogin: login
        }, transaction);

        const {
            certsIds,
            usersData
        } = yield Certificate.getCertsByGlobalUser(globalUserId, trialTemplateIds, transaction);

        yield Certificate.nullify(certsIds, 'ban', transaction);
        yield sendNullifyLetters(usersData, 'banCert');

        this.status = 201;
    },

    *unban() {
        const accessControl = new AdminAccessControl(this.state);

        yield accessControl.hasSupportAccess();

        assert.bySchema(this.request.body, 'Unban');

        const { login, reason, trialTemplateIds, isSuperUnban } = this.request.body;
        const { transaction } = this.state;

        const admin = yield AdminUser.findOrCreate(accessControl.login, this.ip, transaction);
        const storedUser = yield WebUserModel.findUser({ where: { login }, transaction });

        assert(storedUser, 404, 'User not found', 'UNF', { login });
        assert(storedUser.globalUserId, 404, 'Global user not found', 'GNF', { userLogin: login });

        if (isSuperUnban) {
            yield GlobalUser.updateSuperban({
                globalUserId: storedUser.globalUserId,
                isBanned: false
            }, transaction);
        }

        yield Ban.unbanUser({
            globalUserId: storedUser.globalUserId,
            adminId: admin.id,
            reason,
            trialTemplateIds,
            userLogin: login
        }, transaction);

        this.status = 201;
    },

    *associateUsers() {
        const accessControl = new AdminAccessControl(this.state);

        yield accessControl.hasSupportAccess();

        const { logins } = this.request.body || {};

        assert(_.isArray(logins), 400, 'Logins is not an array', 'LNA', { logins });
        assert(_.every(logins, _.isString), 400, 'Login is not a string', 'LNS', { logins });

        const { transaction } = this.state;
        const attributes = ['id', 'login', 'globalUserId'];

        const storedUsers = yield WebUserModel.findByLogins({ logins, attributes, transaction });
        const usersDiff = _.difference(logins, _.map(storedUsers, 'login'));

        assert(_.isEmpty(usersDiff), 404, 'Some users not found', 'SUN', { logins: usersDiff });

        yield GlobalUser.associateUsers(storedUsers, transaction);

        this.status = 204;
    }
};
